/*
 * @(#)SVGInputFormat.java  0.1  November 25, 2006
 *
 * Copyright (c) 1996-2007 by the original authors of JHotDraw
 * and all its contributors ("JHotDraw.org")
 * All rights reserved.
 *
 * This software is the confidential and proprietary information of
 * JHotDraw.org ("Confidential Information"). You shall not disclose
 * such Confidential Information and shall use it only in accordance
 * with the terms of the license agreement you entered into with
 * JHotDraw.org.
 */

package org.jhotdraw.samples.svg.io;

import java.awt.*;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.awt.datatransfer.UnsupportedFlavorException;
import java.awt.geom.*;
import java.awt.image.*;
import java.io.*;
import java.net.*;
import java.util.*;
import javax.imageio.*;
import javax.swing.*;
import javax.swing.text.*;
import net.n3.nanoxml.*;
import org.jhotdraw.draw.*;
import org.jhotdraw.draw.InputFormat;
import org.jhotdraw.xml.css.StyleManager;
import org.jhotdraw.xml.*;
import org.jhotdraw.geom.*;
import org.jhotdraw.io.*;
import org.jhotdraw.samples.svg.*;
import org.jhotdraw.samples.svg.figures.*;
import static org.jhotdraw.samples.svg.SVGConstants.*;
import static org.jhotdraw.samples.svg.SVGAttributeKeys.*;
import org.jhotdraw.xml.css.CSSParser;


/**
 * SVGInputFormat.
 * This format is aimed to comply to the Scalable Vector Graphics (SVG) Tiny 1.2
 * Specification supporting the <code>SVG-static</code> feature string.
 * <a href="http://www.w3.org/TR/SVGMobile12/">http://www.w3.org/TR/SVGMobile12/</a>
 *
 *
 * @author Werner Randelshofer
 * @version 0.1 November 25, 2006 Created (Experimental).
 */
public class SVGInputFormat implements InputFormat {
    private SVGFigureFactory factory;
    private URL url;
    private final static HashMap<String,WindingRule> fillRuleMap;
    static {
        fillRuleMap = new HashMap<String, WindingRule>();
        fillRuleMap.put("nonzero", WindingRule.NON_ZERO);
        fillRuleMap.put("evenodd", WindingRule.EVEN_ODD);
    }
    private final static HashMap<String,Integer> strokeLinecapMap;
    static {
        strokeLinecapMap = new HashMap<String, Integer>();
        strokeLinecapMap.put("butt", BasicStroke.CAP_BUTT);
        strokeLinecapMap.put("round", BasicStroke.CAP_ROUND);
        strokeLinecapMap.put("square", BasicStroke.CAP_SQUARE);
    }
    private final static HashMap<String,Integer> strokeLinejoinMap;
    static {
        strokeLinejoinMap = new HashMap<String, Integer>();
        strokeLinejoinMap.put("miter", BasicStroke.JOIN_MITER);
        strokeLinejoinMap.put("round", BasicStroke.JOIN_ROUND);
        strokeLinejoinMap.put("bevel", BasicStroke.JOIN_BEVEL);
    }
    private final static HashMap<String,Double> absoluteFontSizeMap;
    static {
        absoluteFontSizeMap = new HashMap<String,Double>();
        absoluteFontSizeMap.put("xx-small",6.944444);
        absoluteFontSizeMap.put("x-small",8.3333333);
        absoluteFontSizeMap.put("small", 10d);
        absoluteFontSizeMap.put("medium", 12d);
        absoluteFontSizeMap.put("large", 14.4);
        absoluteFontSizeMap.put("x-large", 17.28);
        absoluteFontSizeMap.put("xx-large",20.736);
    }
    private final static HashMap<String,Double> relativeFontSizeMap;
    static {
        relativeFontSizeMap = new HashMap<String,Double>();
        relativeFontSizeMap.put("larger", 1.2);
        relativeFontSizeMap.put("smaller",0.83333333);
    }
    private final static HashMap<String,TextAnchor> textAnchorMap;
    static {
        textAnchorMap = new HashMap<String, TextAnchor>();
        textAnchorMap.put("start", TextAnchor.START);
        textAnchorMap.put("middle", TextAnchor.MIDDLE);
        textAnchorMap.put("end", TextAnchor.END);
    }
    
    /**
     * Maps to all XML elements that are identified by an xml:id.
     */
    public HashMap<String,IXMLElement> identifiedElements;
    /**
     * Maps to all drawing objects from the XML elements they were created from.
     */
    public HashMap<IXMLElement,Object> elementObjects;
    
    
    /**
     * Each SVG element establishes a new Viewport.
     */
    private static class Viewport {
        /**
         * The width of the Viewport.
         */
        public double width = 640d;
        /**
         * The height of the Viewport.
         */
        public double height = 480d;
        
        /**
         * The viewBox specifies the coordinate system within the Viewport.
         */
        public Rectangle2D.Double viewBox = new Rectangle2D.Double(0d,0d,640d,480d);
        
        /**
         * Factor for percent values relative to Viewport width.
         */
        public double widthPercentFactor = 640d / 100d;
        /**
         * Factor for percent values relative to Viewport height.
         */
        public double heightPercentFactor = 480d / 100d;
        
        /**
         * Factor for number values in the user coordinate system.
         * This is the smaller value of width / viewBox.width and height / viewBox.height.
         */
        public double numberFactor;
        
        /**
         * http://www.w3.org/TR/SVGMobile12/coords.html#PreserveAspectRatioAttribute
         * XXX - use a more sophisticated variable here
         */
        public boolean isPreserveAspectRatio = true;
        
        
        public String toString() {
            return "widthPercentFactor:"+widthPercentFactor+";"+
                    "heightPercentFactor:"+heightPercentFactor+";" +
                    "numberFactor:"+numberFactor;
        }
        
    }
    
    /**
     * Each SVG element creates a new Viewport that we store
     * here.
     */
    private Stack<Viewport> viewportStack;
    
    /**
     * Holds the style manager used for applying cascading style sheet CSS rules
     * to the document.
     */
    private StyleManager styleManager;
    
    /**
     * Holds the figures that are currently being read.
     */
    private LinkedList<Figure> figures;
    
    /**
     * Holds the document that is currently being read.
     */
    private IXMLElement document;
    
    /** Creates a new instance. */
    public SVGInputFormat() {
        this(new DefaultSVGFigureFactory());
    }
    public SVGInputFormat(SVGFigureFactory factory) {
        this.factory = factory;
    }
    
    public LinkedList<Figure> readFigures(InputStream in) throws IOException {
        //long start = System.currentTimeMillis();
        this.figures = new LinkedList<Figure>();
        IXMLParser parser;
        try {
            parser = XMLParserFactory.createDefaultXMLParser();
        } catch (Exception ex) {
            InternalError e = new InternalError("Unable to instantiate NanoXML Parser");
            e.initCause(ex);
            throw e;
        }
        IXMLReader reader = new StdXMLReader(in);
        parser.setReader(reader);
        try {
            document = (IXMLElement) parser.parse();
        } catch (XMLException ex) {
            IOException e = new IOException(ex.getMessage());
            e.initCause(ex);
            throw e;
        }
        
        // Search for the first 'svg' element in the XML document
        // in preorder sequence
        IXMLElement svg = document;
        Stack<Iterator> stack = new Stack<Iterator>();
        LinkedList<IXMLElement> ll = new LinkedList<IXMLElement>();
        ll.add(document);
        stack.push(ll.iterator());
        while (!stack.empty() && stack.peek().hasNext()) {
            Iterator<IXMLElement> iter = stack.peek();
            IXMLElement node = iter.next();
            Iterator<IXMLElement> children = node.getChildren().iterator();
            
            if (! iter.hasNext()) {
                stack.pop();
            }
            if (children.hasNext()) {
                stack.push(children);
            }
            if (node.getName() != null &&
                    node.getName().equals("svg") &&
                    (node.getNamespace() == null ||
                    node.getNamespace().equals(SVG_NAMESPACE))) {
                svg = node;
                break;
            }
        }
        
        
        if (svg.getName() == null ||
                ! svg.getName().equals("svg") ||
                (svg.getNamespace() != null &&
                ! svg.getNamespace().equals(SVG_NAMESPACE))) {
            throw new IOException("'svg' element expected: "+svg.getName());
        }
        //long end1 = System.currentTimeMillis();
        
        // Flatten CSS Styles
        initStorageContext();
        flattenStyles(svg);
        //long end2 = System.currentTimeMillis();
        
        readSVGElement(svg);
        /*long end = System.currentTimeMillis();
        System.out.println("SVGInputFormat elapsed:"+(end-start));
        System.out.println("SVGInputFormat read:"+(end1-start));
        System.out.println("SVGInputFormat flatten:"+(end2-end1));
        System.out.println("SVGInputFormat build:"+(end-end2));
        */
         return figures;
    }
    private void initStorageContext() {
        identifiedElements = new HashMap<String,IXMLElement>();
        elementObjects = new HashMap<IXMLElement,Object>();
        viewportStack = new Stack<Viewport>();
        viewportStack.push(new Viewport());
        styleManager = new StyleManager();
    }
    
    /**
     * Flattens all CSS styles.
     * Styles defined in a "style" attribute and in CSS rules are converted
     * into attributes with the same name.
     */
    private void flattenStyles(IXMLElement elem)
    throws IOException {
        if (elem.getName() != null && elem.getName().equals("style") &&
                readAttribute(elem, "type", "").equals("text/css")) {
            CSSParser cssParser = new CSSParser();
            cssParser.parse(elem.getContent(), styleManager);
        } else {
            
            if (elem.getNamespace() == null ||
                    elem.getNamespace().equals(SVG_NAMESPACE)) {
                
                String style = readAttribute(elem, "style", null);
                if (style != null) {
                    for (String styleProperty : style.split(";"))  {
                        String[] stylePropertyElements = styleProperty.split(":");
                        if (stylePropertyElements.length == 2 &&
                                ! elem.hasAttribute(stylePropertyElements[0].trim(), SVG_NAMESPACE)) {
                            //System.out.println("flatten:"+Arrays.toString(stylePropertyElements));
                            elem.setAttribute(stylePropertyElements[0].trim(), SVG_NAMESPACE, stylePropertyElements[1].trim());
                        }
                    }
                }
                
                styleManager.applyStylesTo(elem);
                
                for (IXMLElement node : elem.getChildren()) {
                    if (node instanceof IXMLElement) {
                        IXMLElement child = (IXMLElement) node;
                        flattenStyles(child);
                    }
                }
            }
        }
    }
    
    
    /**
     * Reads an SVG element of any kind.
     * @return Returns the Figure, if the SVG element represents a Figure.
     * Returns null in all other cases.
     */
    private Figure readElement(IXMLElement elem)
    throws IOException {
        Figure f = null;
        if (elem.getNamespace() == null ||
                elem.getNamespace().equals(SVG_NAMESPACE)) {
            String name = elem.getName();
            if (name.equals("circle")) {
                f = readCircleElement(elem);
            } else if (name.equals("defs")) {
                readDefsElement(elem);
                f = null;
            } else if (name.equals("ellipse")) {
                f = readEllipseElement(elem);
            } else if (name.equals("g")) {
                f = readGElement(elem);
            } else if (name.equals("image")) {
                f = readImageElement(elem);
            } else if (name.equals("line")) {
                f = readLineElement(elem);
            } else if (name.equals("linearGradient")) {
                readLinearGradientElement(elem);
                f = null;
            } else if (name.equals("path")) {
                f = readPathElement(elem);
            } else if (name.equals("polygon")) {
                f = readPolygonElement(elem);
            } else if (name.equals("polyline")) {
                f = readPolylineElement(elem);
            } else if (name.equals("radialGradient")) {
                readRadialGradientElement(elem);
                f = null;
            } else if (name.equals("rect")) {
                f = readRectElement(elem);
            } else if (name.equals("solidColor")) {
                readSolidColorElement(elem);
                f = null;
            } else if (name.equals("svg")) {
                f = readSVGElement(elem);
                //f = readGElement(elem);
            } else if (name.equals("switch")) {
                f = readSwitchElement(elem);
            } else if (name.equals("text")) {
                f = readTextElement(elem);
            } else if (name.equals("title")) {
                //FIXME - Implement reading of title element
                //f = readTitleElement(elem);
            } else if (name.equals("use")) {
                f = readUseElement(elem);
            } else if (name.equals("style")) {
                // Nothing to do, style elements have been already
                // processed in method flattenStyles
            } else {
                System.out.println("SVGInputFormat not implemented for <"+name+">");
            }
        }
        if (f instanceof SVGFigure) {
            if (((SVGFigure) f).isEmpty()) {
                // System.out.println("Empty figure "+f);
                return null;
            }
        } else if (f != null) {
            System.out.println("SVGInputFormat warning: not an SVGFigure "+f);
        }
        
        return f;
    }
    
    /**
     * Reads an SVG "defs" element.
     */
    private void readDefsElement(IXMLElement elem)
    throws IOException {
        for (IXMLElement node : elem.getChildren()) {
            if (node instanceof IXMLElement) {
                IXMLElement child = (IXMLElement) node;
                Figure childFigure = readElement(child);
            }
        }
    }
    
    /**
     * Reads an SVG "g" element.
     */
    private Figure readGElement(IXMLElement elem)
    throws IOException {
        HashMap<AttributeKey,Object> a = new HashMap<AttributeKey,Object>();
        readCoreAttributes(elem, a);
        readTransformAttribute(elem, a);
        AffineTransform transform = (AffineTransform) a.get(TRANSFORM);
        CompositeFigure g = factory.createG(a);
        
        for (IXMLElement node : elem.getChildren()) {
            if (node instanceof IXMLElement) {
                IXMLElement child = (IXMLElement) node;
                Figure childFigure = readElement(child);
                // skip invisible elements
                if (readAttribute(child, "visibility", "visible").equals("visible") &&
                        ! readAttribute(child, "display", "inline").equals("none")) {
                    if (childFigure != null) {
                        g.basicAdd(childFigure);
                    }
                }
            }
        }
        if (transform != null) {
            g.basicTransform(transform);
        }
        return g;
    }
    
    
    
    /**
     * Reads an SVG "svg" element.
     */
    private Figure readSVGElement(IXMLElement elem)
    throws IOException {
        // Establish a new viewport
        Viewport viewport = new Viewport();
        
        viewport.width = toWidth(elem, readAttribute(elem, "width", "100%"));
        viewport.height = toHeight(elem, readAttribute(elem, "height", "100%"));
        
        if (readAttribute(elem, "viewBox", "none").equals("none")) {
            viewport.viewBox.width = viewport.width;
            viewport.viewBox.height = viewport.height;
        } else {
            String[] viewBoxValues = toWSOrCommaSeparatedArray(readAttribute(elem, "viewBox", "none"));
            viewport.viewBox.x = toNumber(elem, viewBoxValues[0]);
            viewport.viewBox.y = toNumber(elem, viewBoxValues[1]);
            viewport.viewBox.width = toNumber(elem, viewBoxValues[2]);
            viewport.viewBox.height = toNumber(elem, viewBoxValues[3]);
        }
        viewport.isPreserveAspectRatio = ! readAttribute(elem, "preserveAspectRatio", "none").equals("none");
        viewport.widthPercentFactor = viewport.viewBox.width / 100d;
        viewport.heightPercentFactor = viewport.viewBox.height / 100d;
        
        viewport.numberFactor = Math.min(
                viewport.width / viewport.viewBox.width,
                viewport.height / viewport.viewBox.height
                );
        
        
        
        AffineTransform viewBoxTransform = new AffineTransform();
        viewBoxTransform.translate(
                -viewport.viewBox.x * viewport.width / viewport.viewBox.width,
                -viewport.viewBox.y *  viewport.height / viewport.viewBox.height
                );
        if (viewport.isPreserveAspectRatio) {
            double factor = Math.min(
                    viewport.width / viewport.viewBox.width,
                    viewport.height / viewport.viewBox.height
                    );
            viewBoxTransform.scale(factor, factor);
        } else {
            viewBoxTransform.scale(
                    viewport.width / viewport.viewBox.width,
                    viewport.height / viewport.viewBox.height
                    );
        }
        viewportStack.push(viewport);
        
        
        // Read the figures
        for (IXMLElement node : elem.getChildren()) {
            if (node instanceof IXMLElement) {
                IXMLElement child = (IXMLElement) node;
                Figure childFigure = readElement(child);
                // skip invisible elements
                if (readAttribute(child, "visibility", "visible").equals("visible") &&
                        ! readAttribute(child, "display", "inline").equals("none")) {
                    
                    if (childFigure != null) {
                        childFigure.basicTransform(viewBoxTransform);
                        figures.add(childFigure);
                    }
                }
            }
        }
        
        viewportStack.pop();
        return null;
    }
    
    /**
     * Reads an SVG "rect" element.
     */
    private Figure readRectElement(IXMLElement elem)
    throws IOException {
        HashMap<AttributeKey,Object> a = new HashMap<AttributeKey,Object>();
        readCoreAttributes(elem, a);
        readTransformAttribute(elem, a);
        readShapeAttributes(elem, a);
        
        double x = toNumber(elem, readAttribute(elem, "x", "0"));
        double y = toNumber(elem, readAttribute(elem, "y", "0"));
        double w = toWidth(elem, readAttribute(elem, "width", "0"));
        double h = toHeight(elem, readAttribute(elem, "height", "0"));
        
        String rxValue = readAttribute(elem, "rx", "none");
        String ryValue = readAttribute(elem, "ry", "none");
        if (rxValue.equals("none")) {
            rxValue = ryValue;
        }
        if (ryValue.equals("none")) {
            ryValue = rxValue;
        }
        double rx = toNumber(elem, rxValue.equals("none") ? "0" : rxValue);
        double ry = toNumber(elem, ryValue.equals("none") ? "0" : ryValue);
        
        Figure figure = factory.createRect(x, y, w, h, rx, ry, a);
        elementObjects.put(elem, figure);
        return figure;
    }
    /**
     * Reads an SVG "circle" element.
     */
    private Figure readCircleElement(IXMLElement elem)
    throws IOException {
        HashMap<AttributeKey,Object> a = new HashMap<AttributeKey,Object>();
        readCoreAttributes(elem, a);
        readTransformAttribute(elem, a);
        readShapeAttributes(elem, a);
        
        double cx = toNumber(elem, readAttribute(elem, "cx", "0"));
        double cy = toNumber(elem, readAttribute(elem, "cy", "0"));
        double r = toNumber(elem, readAttribute(elem, "r", "0"));
        
        Figure figure = factory.createCircle(cx, cy, r, a);
        elementObjects.put(elem, figure);
        return figure;
    }
    /**
     * Reads an SVG "ellipse" element.
     */
    private Figure readEllipseElement(IXMLElement elem)
    throws IOException {
        HashMap<AttributeKey,Object> a = new HashMap<AttributeKey,Object>();
        readCoreAttributes(elem, a);
        readTransformAttribute(elem, a);
        readShapeAttributes(elem, a);
        
        double cx = toNumber(elem, readAttribute(elem, "cx", "0"));
        double cy = toNumber(elem, readAttribute(elem, "cy", "0"));
        double rx = toNumber(elem, readAttribute(elem, "rx", "0"));
        double ry = toNumber(elem, readAttribute(elem, "ry", "0"));
        
        Figure figure = factory.createEllipse(cx, cy, rx, ry, a);
        elementObjects.put(elem, figure);
        return figure;
    }
    /**
     * Reads an SVG "image" element.
     */
    private Figure readImageElement(IXMLElement elem)
    throws IOException {
        HashMap<AttributeKey,Object> a = new HashMap<AttributeKey,Object>();
        readCoreAttributes(elem, a);
        readTransformAttribute(elem, a);
        readImageAttributes(elem, a);
        
        double x = toNumber(elem, readAttribute(elem, "x", "0"));
        double y = toNumber(elem, readAttribute(elem, "y", "0"));
        double w = toWidth(elem, readAttribute(elem, "width", "0"));
        double h = toHeight(elem, readAttribute(elem, "height", "0"));
        
        String href = readAttribute(elem, "xlink:href",null);
        if (href == null) {
            href = readAttribute(elem, "href",null);
        }
        byte[] imageData = null;
        if (href != null) {
            if (href.startsWith("data:")) {
                int semicolonPos = href.indexOf(';');
                if (semicolonPos != -1) {
                    if (href.indexOf(";base64,") == semicolonPos) {
                        imageData = Base64.decode(href.substring(semicolonPos+8));
                    } else {
                        throw new IOException("Unsupported encoding in data href in image element:"+href);
                    }
                } else {
                    throw new IOException("Unsupported data href in image element:"+href);
                }
            } else {
                URL imageUrl = new URL(url, href);
                
                // Read the image data from the URL into a byte array
                ByteArrayOutputStream bout = new ByteArrayOutputStream();
                byte[] buf = new byte[512];
                int len = 0;
                InputStream in = null;
                try {
                    in = imageUrl.openStream();
                    while ((len = in.read(buf)) > 0) {
                        bout.write(buf, 0, len);
                    }
                } finally {
                    if (in != null) { in.close(); }
                }
                imageData = bout.toByteArray();
            }
        }
        // Create a buffered image from the image data
        BufferedImage bufferedImage = null;
        if (imageData != null) {
            bufferedImage = ImageIO.read(new ByteArrayInputStream(imageData));
        }
        // Delete the image data in case of failure
        if (bufferedImage == null) {
            imageData = null;
            //System.out.println("FAILED:"+imageUrl);
        }
        
        // Create a figure from the image data and the buffered image.
        Figure figure = factory.createImage(x, y, w, h, imageData, bufferedImage, a);
        elementObjects.put(elem, figure);
        return figure;
    }
    /**
     * Reads an SVG "line" element.
     */
    private Figure readLineElement(IXMLElement elem)
    throws IOException {
        HashMap<AttributeKey,Object> a = new HashMap<AttributeKey,Object>();
        readCoreAttributes(elem, a);
        readTransformAttribute(elem, a);
        readShapeAttributes(elem, a);
        
        double x1 = toNumber(elem, readAttribute(elem, "x1", "0"));
        double y1 = toNumber(elem, readAttribute(elem, "y1", "0"));
        double x2 = toNumber(elem, readAttribute(elem, "x2", "0"));
        double y2 = toNumber(elem, readAttribute(elem, "y2", "0"));
        
        Figure figure = factory.createLine(x1, y1, x2, y2, a);
        elementObjects.put(elem, figure);
        return figure;
    }
    /**
     * Reads an SVG "polyline" element.
     */
    private Figure readPolylineElement(IXMLElement elem)
    throws IOException {
        HashMap<AttributeKey,Object> a = new HashMap<AttributeKey,Object>();
        readCoreAttributes(elem, a);
        readTransformAttribute(elem, a);
        readShapeAttributes(elem, a);
        
        Point2D.Double[] points = toPoints(elem, readAttribute(elem, "points", ""));
        
        Figure figure = factory.createPolyline(points, a);
        elementObjects.put(elem, figure);
        return figure;
    }
    /**
     * Reads an SVG "polygon" element.
     */
    private Figure readPolygonElement(IXMLElement elem)
    throws IOException {
        HashMap<AttributeKey,Object> a = new HashMap<AttributeKey,Object>();
        readCoreAttributes(elem, a);
        readTransformAttribute(elem, a);
        readShapeAttributes(elem, a);
        
        Point2D.Double[] points = toPoints(elem, readAttribute(elem, "points", ""));
        
        Figure figure = factory.createPolygon(points, a);
        elementObjects.put(elem, figure);
        return figure;
    }
    /**
     * Reads an SVG "path" element.
     */
    private Figure readPathElement(IXMLElement elem)
    throws IOException {
        HashMap<AttributeKey,Object> a = new HashMap<AttributeKey,Object>();
        readCoreAttributes(elem, a);
        readTransformAttribute(elem, a);
        readShapeAttributes(elem, a);
        
        BezierPath[] beziers = toPath(elem, readAttribute(elem, "d", ""));
        
        Figure figure = factory.createPath(beziers, a);
        elementObjects.put(elem, figure);
        return figure;
    }
    /**
     * Reads an SVG "text" element.
     */
    private Figure readTextElement(IXMLElement elem)
    throws IOException {
        HashMap<AttributeKey,Object> a = new HashMap<AttributeKey,Object>();
        readCoreAttributes(elem, a);
        readTransformAttribute(elem, a);
        readShapeAttributes(elem, a);
        readFontAttributes(elem, a);
        readTextAttributes(elem, a);
        
        String[] xStr = toCommaSeparatedArray(readAttribute(elem, "x", "0"));
        String[] yStr = toCommaSeparatedArray(readAttribute(elem, "y", "0"));
        
        Point2D.Double[] coordinates = new Point2D.Double[Math.max(xStr.length, yStr.length)];
        double lastX = 0;
        double lastY = 0;
        for (int i=0; i < coordinates.length; i++) {
            if (xStr.length > i) {
                try {
                    lastX = toNumber(elem, xStr[i]);
                } catch (NumberFormatException ex) {
                }
            }
            if (yStr.length > i) {
                try {
                    lastY = toNumber(elem, yStr[i]);
                } catch (NumberFormatException ex) {
                }
            }
            coordinates[i] = new Point2D.Double(lastX, lastY);
        }
        
        
        String[] rotateStr = toCommaSeparatedArray(readAttribute(elem, "rotate", ""));
        double[] rotate = new double[rotateStr.length];
        for (int i=0; i < rotateStr.length; i++) {
            try {
                rotate[i] = toDouble(elem, rotateStr[i]);
            } catch (NumberFormatException ex) {
                rotate[i] = 0;
            }
        }
        
        DefaultStyledDocument doc = new DefaultStyledDocument();
        
        try {
            if (elem.getContent() != null) {
                doc.insertString(0, toText(elem, elem.getContent()), null);
            } else {
                for (IXMLElement node : elem.getChildren()) {
                    if (node.getName() == null) {
                        doc.insertString(0, toText(elem, node.getContent()), null);
                    } else if (node.getName().equals("tspan")) {
                        readTSpanElement((IXMLElement) node, doc);
                    } else {
                        System.out.println("SVGInputFormat unsupported text node <"+node.getName()+">");
                    }
                }
            }
        } catch (BadLocationException e) {
            InternalError ex = new InternalError(e.getMessage());
            ex.initCause(e);
            throw ex;
        }
        Figure figure = factory.createText(coordinates, rotate, doc, a);
        elementObjects.put(elem, figure);
        return figure;
    }
    /**
     * Reads an SVG "textArea" element.
     */
    private Figure readTextAreaElement(IXMLElement elem)
    throws IOException {
        HashMap<AttributeKey,Object> a = new HashMap<AttributeKey,Object>();
        readCoreAttributes(elem, a);
        readTransformAttribute(elem, a);
        readShapeAttributes(elem, a);
        readFontAttributes(elem, a);
        readTextAttributes(elem, a);
        readTextFlowAttributes(elem, a);
        
        double x = toNumber(elem, readAttribute(elem, "x", "0"));
        double y = toNumber(elem, readAttribute(elem, "y", "0"));
// XXX - Handle "auto" width and height
        double w = toWidth(elem, readAttribute(elem, "width", "0"));
        double h = toHeight(elem, readAttribute(elem, "height", "0"));
        
        DefaultStyledDocument doc = new DefaultStyledDocument();
        
        try {
            if (elem.getContent() != null) {
                doc.insertString(0, toText(elem, elem.getContent()), null);
            } else {
                for (IXMLElement node : elem.getChildren()) {
                    if (node.getName() == null) {
                        doc.insertString(0, toText(elem, node.getContent()), null);
                    } else if (node.getName().equals("tspan")) {
                        readTSpanElement((IXMLElement) node, doc);
                    } else {
                        System.out.println("SVGInputFormat unknown  text node "+node.getName());
                    }
                }
            }
        } catch (BadLocationException e) {
            InternalError ex = new InternalError(e.getMessage());
            ex.initCause(e);
            throw ex;
        }
        
        Figure figure = factory.createTextArea(x, y, w, h, doc, a);
        elementObjects.put(elem, figure);
        return figure;
    }
    /**
     * Reads an SVG "tspan" element.
     */
    private void readTSpanElement(IXMLElement elem, DefaultStyledDocument doc)
    throws IOException {
        try {
            if (elem.getContent() != null) {
                doc.insertString(doc.getLength(), toText(elem, elem.getContent()), null);
            } else {
                for (IXMLElement node : elem.getChildren()) {
                    if (node instanceof IXMLElement) {
                        IXMLElement child = (IXMLElement) node;
                        if (node.getName() != null && node.getName().equals("tspan")) {
                            readTSpanElement((IXMLElement) node, doc);
                        } else {
                            System.out.println("SVGInputFormat unknown text node "+node.getName());
                        }
                    } else {
                        if (node.getName() == null) {
                            doc.insertString(doc.getLength(), toText(elem, node.getContent()), null);
                        }                    }
                }
            }
        } catch (BadLocationException e) {
            InternalError ex = new InternalError(e.getMessage());
            ex.initCause(e);
            throw ex;
        }
    }
    
    private final static HashSet<String> supportedFeatures = new HashSet<String>(
            Arrays.asList(new String[] {
        "http://www.w3.org/Graphics/SVG/feature/1.2/#SVG-static",
        //"http://www.w3.org/Graphics/SVG/feature/1.2/#SVG-static-DOM",
        //"http://www.w3.org/Graphics/SVG/feature/1.2/#SVG-animated",
        //"http://www.w3.org/Graphics/SVG/feature/1.2/#SVG-all",
        "http://www.w3.org/Graphics/SVG/feature/1.2/#CoreAttribute",
        //"http://www.w3.org/Graphics/SVG/feature/1.2/#NavigationAttribute",
        "http://www.w3.org/Graphics/SVG/feature/1.2/#Structure",
        "http://www.w3.org/Graphics/SVG/feature/1.2/#ConditionalProcessing",
        "http://www.w3.org/Graphics/SVG/feature/1.2/#ConditionalProcessingAttribute",
        "http://www.w3.org/Graphics/SVG/feature/1.2/#Image",
        //"http://www.w3.org/Graphics/SVG/feature/1.2/#Prefetch",
        //"http://www.w3.org/Graphics/SVG/feature/1.2/#Discard",
        "http://www.w3.org/Graphics/SVG/feature/1.2/#Shape",
        "http://www.w3.org/Graphics/SVG/feature/1.2/#Text",
        "http://www.w3.org/Graphics/SVG/feature/1.2/#PaintAttribute",
        "http://www.w3.org/Graphics/SVG/feature/1.2/#OpacityAttribute",
        "http://www.w3.org/Graphics/SVG/feature/1.2/#GraphicsAttribute",
        "http://www.w3.org/Graphics/SVG/feature/1.2/#Gradient",
        "http://www.w3.org/Graphics/SVG/feature/1.2/#SolidColor",
        //"http://www.w3.org/Graphics/SVG/feature/1.2/#Hyperlinking",
        //"http://www.w3.org/Graphics/SVG/feature/1.2/#XlinkAttribute",
        //"http://www.w3.org/Graphics/SVG/feature/1.2/#ExternalResourcesRequired",
        //"http://www.w3.org/Graphics/SVG/feature/1.2/#Scripting",
        //"http://www.w3.org/Graphics/SVG/feature/1.2/#Handler",
        //"http://www.w3.org/Graphics/SVG/feature/1.2/#Listener",
        //"http://www.w3.org/Graphics/SVG/feature/1.2/#TimedAnimation",
        //"http://www.w3.org/Graphics/SVG/feature/1.2/#Animation",
        //"http://www.w3.org/Graphics/SVG/feature/1.2/#Audio",
        //"http://www.w3.org/Graphics/SVG/feature/1.2/#Video",
        "http://www.w3.org/Graphics/SVG/feature/1.2/#Font",
        //"http://www.w3.org/Graphics/SVG/feature/1.2/#Extensibility",
        //"http://www.w3.org/Graphics/SVG/feature/1.2/#MediaAttribute",
        //"http://www.w3.org/Graphics/SVG/feature/1.2/#TextFlow",
        //"http://www.w3.org/Graphics/SVG/feature/1.2/#TransformedVideo",
        //"http://www.w3.org/Graphics/SVG/feature/1.2/#ComposedVideo",
    }));
    
    /**
     * Evaluates an SVG "switch" element.
     *
     */
    private Figure readSwitchElement(IXMLElement elem)
    throws IOException {
        for (IXMLElement node : elem.getChildren()) {
            if (node instanceof IXMLElement) {
                IXMLElement child = (IXMLElement) node;
                String[] requiredFeatures = toWSOrCommaSeparatedArray(readAttribute(child, "requiredFeatures", ""));
                String[] requiredExtensions = toWSOrCommaSeparatedArray(readAttribute(child, "requiredExtensions", ""));
                String[] systemLanguage = toWSOrCommaSeparatedArray(readAttribute(child, "systemLanguage", ""));
                String[] requiredFormats = toWSOrCommaSeparatedArray(readAttribute(child, "requiredFormats", ""));
                String[] requiredFonts = toWSOrCommaSeparatedArray(readAttribute(child, "requiredFonts", ""));
                
                boolean isMatch;
                
                isMatch = supportedFeatures.containsAll(Arrays.asList(requiredFeatures)) &&
                        requiredExtensions.length == 0 &&
                        requiredFormats.length == 0 &&
                        requiredFonts.length == 0;
                
                if (isMatch && systemLanguage.length > 0) {
                    isMatch = false;
                    Locale locale = Locale.getDefault();
                    for (String lng : systemLanguage) {
                        int p = lng.indexOf('-');
                        if (p == -1) {
                            if (locale.getLanguage().equals(lng)) {
                                isMatch = true;
                                break;
                            }
                        } else {
                            if (locale.getLanguage().equals(lng.substring(0, p)) &&
                                    locale.getCountry().toLowerCase().equals(lng.substring(p + 1))) {
                                isMatch = true;
                                break;
                            }
                        }
                    }
                }
                if (isMatch) {
                    Figure figure = readElement(child);
                    if (readAttribute(child, "visibility", "visible").equals("visible") &&
                            ! readAttribute(child, "display", "inline").equals("none")) {
                        return figure;
                    } else {
                        return null;
                    }
                }
            }
        }
        return null;
    }
    /**
     * Reads an SVG "use" element.
     */
    private Figure readUseElement(IXMLElement elem)
    throws IOException {
        HashMap<AttributeKey,Object> a = new HashMap<AttributeKey,Object>();
        readCoreAttributes(elem, a);
        HashMap<AttributeKey,Object> a2 = new HashMap<AttributeKey,Object>();
        readTransformAttribute(elem, a2);
        readUseShapeAttributes(elem, a2);
        
        String href = readAttribute(elem, "xlink:href",null);
        if (href != null && href.startsWith("#")) {
            
            IXMLElement refElem = identifiedElements.get(href.substring(1));
            if (refElem == null) {
                System.out.println("SVGInputFormat couldn't find href for <use> element:"+href);
            } else {
                Object obj = readElement(refElem);
                if (obj instanceof Figure) {
                    Figure figure = (Figure) ((Figure) obj).clone();
                    for (Map.Entry<AttributeKey, Object> entry : a2.entrySet()) {
                        //System.out.println(entry);
                        //if (! figure.getAttributes().containsKey(entry.getKey())) {
                        figure.basicSetAttribute(entry.getKey(), entry.getValue());
                        //}
                    }
                    
                    
                    AffineTransform tx = new AffineTransform();
                    double x = toNumber(elem, readAttribute(elem, "x", "0"));
                    double y = toNumber(elem, readAttribute(elem, "y", "0"));
                    tx.translate(x, y);
                    figure.basicTransform(tx);
                    
                    return figure;
                }
            }
        }
        return null;
    }
    /**
     * Reads an attribute that is inherited.
     */
    private String readInheritAttribute(IXMLElement elem, String attributeName, String defaultValue) {
        if (elem.hasAttribute(attributeName, SVG_NAMESPACE)) {
            String value = elem.getAttribute(attributeName, SVG_NAMESPACE, null);
            if (value.equals("inherit")) {
                return readInheritAttribute(elem.getParent(), attributeName, defaultValue);
            } else {
                return value;
            }
        } else if (elem.hasAttribute(attributeName)) {
            String value = elem.getAttribute(attributeName);
            if (value.equals("inherit")) {
                return readInheritAttribute(elem.getParent(), attributeName, defaultValue);
            } else {
                return value;
            }
        } else if (elem.getParent() != null &&
                (elem.getParent().getNamespace() == null ||
                elem.getParent().getNamespace().equals(SVG_NAMESPACE))) {
            return readInheritAttribute(elem.getParent(), attributeName, defaultValue);
        } else {
            return defaultValue;
        }
    }
    /**
     * Reads a font size attribute that is inherited.
     * As specified by
     * http://www.w3.org/TR/SVGMobile12/text.html#FontPropertiesUsedBySVG
     * http://www.w3.org/TR/2006/CR-xsl11-20060220/#font-size
     */
    private double readInheritFontSizeAttribute(IXMLElement elem, String attributeName, String defaultValue)
    throws IOException {
        String value = null;
        if (elem.hasAttribute(attributeName, SVG_NAMESPACE)) {
            value = elem.getAttribute(attributeName, SVG_NAMESPACE, null);
        } else if (elem.hasAttribute(attributeName)) {
            value = elem.getAttribute(attributeName, null);
        } else {
            value = defaultValue;
        }
        
        if (value.equals("inherit")) {
            return readInheritFontSizeAttribute(elem.getParent(), attributeName, defaultValue);
        } else if (absoluteFontSizeMap.containsKey(value)) {
            return absoluteFontSizeMap.get(value);
        } else if (relativeFontSizeMap.containsKey(value)) {
            return relativeFontSizeMap.get(value) * readInheritFontSizeAttribute(elem.getParent(), attributeName, defaultValue);
        } else if (value.endsWith("%")) {
            double factor = Double.valueOf(value.substring(0, value.length() - 1));
            return factor * readInheritFontSizeAttribute(elem.getParent(), attributeName, defaultValue);
        } else {
            //return toScaledNumber(elem, value);
            return toNumber(elem, value);
        }
        
    }
    
    /**
     * Reads an attribute that is not inherited, unless its value is "inherit".
     */
    private String readAttribute(IXMLElement elem, String attributeName, String defaultValue) {
        if (elem.hasAttribute(attributeName, SVG_NAMESPACE)) {
            String value = elem.getAttribute(attributeName, SVG_NAMESPACE, null);
            if (value.equals("inherit")) {
                return readAttribute(elem.getParent(), attributeName, defaultValue);
            } else {
                return value;
            }
        } else if (elem.hasAttribute(attributeName) ) {
            String value = elem.getAttribute(attributeName, null);
            if (value.equals("inherit")) {
                return readAttribute(elem.getParent(), attributeName, defaultValue);
            } else {
                return value;
            }
        } else {
            return defaultValue;
        }
    }
    
    /**
     * Returns a value as a width.
     * http://www.w3.org/TR/SVGMobile12/types.html#DataTypeLength
     */
    private double toWidth(IXMLElement elem, String str) throws IOException {
        // XXX - Compute xPercentFactor from viewport
        return toLength(elem, str,
                viewportStack.peek().widthPercentFactor);
    }
    /**
     * Returns a value as a height.
     * http://www.w3.org/TR/SVGMobile12/types.html#DataTypeLength
     */
    private double toHeight(IXMLElement elem, String str) throws IOException {
        // XXX - Compute yPercentFactor from viewport
        return toLength(elem, str,
                viewportStack.peek().heightPercentFactor);
    }
    /**
     * Returns a value as a number.
     * http://www.w3.org/TR/SVGMobile12/types.html#DataTypeNumber
     */
    private double toNumber(IXMLElement elem, String str) throws IOException {
        return toLength(elem, str, Math.min(
                viewportStack.peek().widthPercentFactor,
                viewportStack.peek().heightPercentFactor
                ));
    }
    /**
     * Returns a value as a scaled number.
     * http://www.w3.org/TR/SVGMobile12/types.html#DataTypeNumber
     */
    private double toScaledNumber(IXMLElement elem, String str) throws IOException {
        return toLength(elem, str, Math.min(
                viewportStack.peek().widthPercentFactor,
                viewportStack.peek().heightPercentFactor
                )) * viewportStack.peek().numberFactor;
    }
    /**
     * Returns a value as a length.
     * http://www.w3.org/TR/SVGMobile12/types.html#DataTypeLength
     */
    private double toLength(IXMLElement elem, String str, double percentFactor) throws IOException {
        double scaleFactor = 1d;
        if (str == null || str.length() == 0) {
            return 0d;
        }
        
        if (str.endsWith("%")) {
            str = str.substring(0, str.length() - 1);
            scaleFactor = percentFactor;
        } else if (str.endsWith("px")) {
            str = str.substring(0, str.length() - 2);
        } else if (str.endsWith("pt")) {
            str = str.substring(0, str.length() - 2);
            scaleFactor = 1.25;
        } else if (str.endsWith("pc")) {
            str = str.substring(0, str.length() - 2);
            scaleFactor = 15;
        } else if (str.endsWith("mm")) {
            str = str.substring(0, str.length() - 2);
            scaleFactor = 3.543307;
        } else if (str.endsWith("cm")) {
            str = str.substring(0, str.length() - 2);
            scaleFactor = 35.43307;
        } else if (str.endsWith("in")) {
            str = str.substring(0, str.length() - 2);
            scaleFactor = 90;
        } else if (str.endsWith("em")) {
            str = str.substring(0, str.length() - 2);
            // XXX - This doesn't work
            scaleFactor = toLength(elem, readAttribute(elem, "font-size", "0"), percentFactor);
        } else {
            scaleFactor = 1d;
        }
        
        return Double.parseDouble(str) * scaleFactor;
    }
    /**
     * Returns a value as a String array.
     * The values are separated by commas with optional white space.
     */
    public static String[] toCommaSeparatedArray(String str) throws IOException {
        return str.split("\\s*,\\s*");
    }
    /**
     * Returns a value as a String array.
     * The values are separated by whitespace or by commas with optional white
     * space.
     */
    public static String[] toWSOrCommaSeparatedArray(String str) throws IOException {
        String[] result = str.split("(\\s*,\\s*|\\s+)");
        if (result.length == 1 && result[0].equals("")) {
            return new String[0];
        } else {
            return result;
        }
    }
    /**
     * Returns a value as a Point2D.Double array.
     * as specified in http://www.w3.org/TR/SVGMobile12/shapes.html#PointsBNF
     */
    private Point2D.Double[] toPoints(IXMLElement elem, String str) throws IOException {
        
        StringTokenizer tt = new StringTokenizer(str," ,");
        Point2D.Double[] points =new Point2D.Double[tt.countTokens() / 2];
        for (int i=0; i < points.length; i++) {
            
            points[i] = new Point2D.Double(
                    toNumber(elem, tt.nextToken()),
                    toNumber(elem, tt.nextToken())
                    );
        }
        return points;
    }
    
    /**
     * Returns a value as a BezierPath array.
     * as specified in http://www.w3.org/TR/SVGMobile12/shapes.html#PointsBNF
     *
     * Also supports elliptical arc commands 'a' and 'A' as specified in
     * http://www.w3.org/TR/SVG/paths.html#PathDataEllipticalArcCommands
     */
    private BezierPath[] toPath(IXMLElement elem, String str) throws IOException {
        LinkedList<BezierPath> paths = new LinkedList<BezierPath>();
        
        BezierPath path = null;
        Point2D.Double p = new Point2D.Double();
        Point2D.Double c1 = new Point2D.Double();
        Point2D.Double c2 = new Point2D.Double();
        
        StreamTokenizer tt = new StreamTokenizer(new StringReader(str));
        tt.resetSyntax();
        tt.parseNumbers();
        tt.whitespaceChars(0, ' ');
        tt.whitespaceChars(',',',');
        
        
        char nextCommand = 'M';
        char command = 'M';
        Commands: while (tt.nextToken() != StreamTokenizer.TT_EOF) {
            if (tt.ttype > 0) {
                command = (char) tt.ttype;
            } else {
                command = nextCommand;
                tt.pushBack();
            }
            
            BezierPath.Node node;
            
            switch (command) {
                case 'M' :
                    // absolute-moveto x y
                    if (path != null) {
                        paths.add(path);
                    }
                    path = new BezierPath();
                    
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("x coordinate missing for 'M' in "+str);
                    }
                    p.x = tt.nval;
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("y coordinate missing for 'M' in "+str);
                    }
                    p.y = tt.nval;
                    path.moveTo(p.x, p.y);
                    nextCommand = 'L';
                    break;
                case 'm' :
                    // relative-moveto dx dy
                    if (path != null) {
                        paths.add(path);
                    }
                    path = new BezierPath();
                    
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("dx coordinate missing for 'm' in "+str);
                    }
                    p.x += tt.nval;
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("dy coordinate missing for 'm' in "+str);
                    }
                    p.y += tt.nval;
                    path.moveTo(p.x, p.y);
                    nextCommand = 'l';
                    
                    break;
                case 'Z' :
                case 'z' :
                    // close path
                    p.x = path.get(0).x[0];
                    p.y = path.get(0).y[0];
                    path.setClosed(true);
                    
                    break;
                case 'L' :
                    // absolute-lineto x y
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("x coordinate missing for 'L' in "+str);
                    }
                    p.x = tt.nval;
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("y coordinate missing for 'L' in "+str);
                    }
                    p.y = tt.nval;
                    path.lineTo(p.x, p.y);
                    nextCommand = 'L';
                    
                    break;
                case 'l' :
                    // relative-lineto dx dy
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("dx coordinate missing for 'l' in "+str);
                    }
                    p.x += tt.nval;
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("dy coordinate missing for 'l' in "+str);
                    }
                    p.y += tt.nval;
                    path.lineTo(p.x, p.y);
                    nextCommand = 'l';
                    
                    break;
                case 'H' :
                    // absolute-horizontal-lineto x
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("x coordinate missing for 'H' in "+str);
                    }
                    p.x = tt.nval;
                    path.lineTo(p.x, p.y);
                    nextCommand = 'H';
                    
                    break;
                case 'h' :
                    // relative-horizontal-lineto dx
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("dx coordinate missing for 'h' in "+str);
                    }
                    p.x += tt.nval;
                    path.lineTo(p.x, p.y);
                    nextCommand = 'h';
                    
                    break;
                case 'V' :
                    // absolute-vertical-lineto y
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("y coordinate missing for 'V' in "+str);
                    }
                    p.y = tt.nval;
                    path.lineTo(p.x, p.y);
                    nextCommand = 'V';
                    
                    break;
                case 'v' :
                    // relative-vertical-lineto dy
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("dy coordinate missing for 'v' in "+str);
                    }
                    p.y += tt.nval;
                    path.lineTo(p.x, p.y);
                    nextCommand = 'v';
                    
                    break;
                case 'C' :
                    // absolute-curveto x1 y1 x2 y2 x y
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("x1 coordinate missing for 'C' in "+str);
                    }
                    c1.x = tt.nval;
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("y1 coordinate missing for 'C' in "+str);
                    }
                    c1.y = tt.nval;
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("x2 coordinate missing for 'C' in "+str);
                    }
                    c2.x = tt.nval;
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("y2 coordinate missing for 'C' in "+str);
                    }
                    c2.y = tt.nval;
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("x coordinate missing for 'C' in "+str);
                    }
                    p.x = tt.nval;
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("y coordinate missing for 'C' in "+str);
                    }
                    p.y = tt.nval;
                    path.curveTo(c1.x, c1.y, c2.x, c2.y, p.x, p.y);
                    nextCommand = 'C';
                    break;
                    
                case 'c' :
                    // relative-curveto dx1 dy1 dx2 dy2 dx dy
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("dx1 coordinate missing for 'c' in "+str);
                    }
                    c1.x = p.x + tt.nval;
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("dy1 coordinate missing for 'c' in "+str);
                    }
                    c1.y = p.y + tt.nval;
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("dx2 coordinate missing for 'c' in "+str);
                    }
                    c2.x = p.x + tt.nval;
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("dy2 coordinate missing for 'c' in "+str);
                    }
                    c2.y = p.y + tt.nval;
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("dx coordinate missing for 'c' in "+str);
                    }
                    p.x += tt.nval;
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("dy coordinate missing for 'c' in "+str);
                    }
                    p.y += tt.nval;
                    path.curveTo(c1.x, c1.y, c2.x, c2.y, p.x, p.y);
                    nextCommand = 'c';
                    break;
                    
                case 'S' :
                    // absolute-shorthand-curveto x2 y2 x y
                    node = path.get(path.size() - 1);
                    c1.x = node.x[0] * 2d - node.x[1];
                    c1.y = node.y[0] * 2d - node.y[1];
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("x2 coordinate missing for 'S' in "+str);
                    }
                    c2.x = tt.nval;
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("y2 coordinate missing for 'S' in "+str);
                    }
                    c2.y = tt.nval;
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("x coordinate missing for 'S' in "+str);
                    }
                    p.x = tt.nval;
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("y coordinate missing for 'S' in "+str);
                    }
                    p.y = tt.nval;
                    path.curveTo(c1.x, c1.y, c2.x, c2.y, p.x, p.y);
                    nextCommand = 'S';
                    break;
                    
                case 's' :
                    // relative-shorthand-curveto dx2 dy2 dx dy
                    node = path.get(path.size() - 1);
                    c1.x = node.x[0] * 2d - node.x[1];
                    c1.y = node.y[0] * 2d - node.y[1];
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("dx2 coordinate missing for 's' in "+str);
                    }
                    c2.x = p.x + tt.nval;
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("dy2 coordinate missing for 's' in "+str);
                    }
                    c2.y = p.y + tt.nval;
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("dx coordinate missing for 's' in "+str);
                    }
                    p.x += tt.nval;
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("dy coordinate missing for 's' in "+str);
                    }
                    p.y += tt.nval;
                    path.curveTo(c1.x, c1.y, c2.x, c2.y, p.x, p.y);
                    nextCommand = 's';
                    break;
                    
                case 'Q' :
                    // absolute-quadto x1 y1 x y
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("x1 coordinate missing for 'Q' in "+str);
                    }
                    c1.x = tt.nval;
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("y1 coordinate missing for 'Q' in "+str);
                    }
                    c1.y = tt.nval;
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("x coordinate missing for 'Q' in "+str);
                    }
                    p.x = tt.nval;
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("y coordinate missing for 'Q' in "+str);
                    }
                    p.y = tt.nval;
                    path.quadTo(c1.x, c1.y, p.x, p.y);
                    nextCommand = 'Q';
                    
                    break;
                    
                case 'q' :
                    // relative-quadto dx1 dy1 dx dy
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("dx1 coordinate missing for 'q' in "+str);
                    }
                    c1.x = p.x + tt.nval;
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("dy1 coordinate missing for 'q' in "+str);
                    }
                    c1.y = p.y + tt.nval;
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("dx coordinate missing for 'q' in "+str);
                    }
                    p.x += tt.nval;
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("dy coordinate missing for 'q' in "+str);
                    }
                    p.y += tt.nval;
                    path.quadTo(c1.x, c1.y, p.x, p.y);
                    nextCommand = 'q';
                    
                    break;
                case 'T' :
                    // absolute-shorthand-quadto x y
                    node = path.get(path.size() - 1);
                    c1.x = node.x[0] * 2d - node.x[1];
                    c1.y = node.y[0] * 2d - node.y[1];
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("x coordinate missing for 'T' in "+str);
                    }
                    p.x = tt.nval;
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("y coordinate missing for 'T' in "+str);
                    }
                    p.y = tt.nval;
                    path.quadTo(c1.x, c1.y, p.x, p.y);
                    nextCommand = 'T';
                    
                    break;
                    
                case 't' :
                    // relative-shorthand-quadto dx dy
                    node = path.get(path.size() - 1);
                    c1.x = node.x[0] * 2d - node.x[1];
                    c1.y = node.y[0] * 2d - node.y[1];
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("dx coordinate missing for 't' in "+str);
                    }
                    p.x += tt.nval;
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("dy coordinate missing for 't' in "+str);
                    }
                    p.y += tt.nval;
                    path.quadTo(c1.x, c1.y, p.x, p.y);
                    nextCommand = 's';
                    
                    break;
                    
                    
                case 'A' : {
                    // absolute-elliptical-arc rx ry x-axis-rotation large-arc-flag sweep-flag x y
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("rx coordinate missing for 'A' in "+str);
                    }
                    // If rX or rY have negative signs, these are dropped;
                    // the absolute value is used instead.
                    double rx = tt.nval;
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("ry coordinate missing for 'A' in "+str);
                    }
                    double ry = tt.nval;
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("x-axis-rotation missing for 'A' in "+str);
                    }
                    double xAxisRotation = tt.nval;
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("large-arc-flag missing for 'A' in "+str);
                    }
                    boolean largeArcFlag = tt.nval != 0;
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("sweep-flag missing for 'A' in "+str);
                    }
                    boolean sweepFlag = tt.nval != 0;
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("x coordinate missing for 'A' in "+str);
                    }
                    p.x = tt.nval;
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("y coordinate missing for 'A' in "+str);
                    }
                    p.y = tt.nval;
                    
                    path.arcTo(rx, ry, xAxisRotation, largeArcFlag, sweepFlag, p.x, p.y);
                    
                    nextCommand = 'A';
                    break;
                    }
                case 'a' : {
                    // absolute-elliptical-arc rx ry x-axis-rotation large-arc-flag sweep-flag x y
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("rx coordinate missing for 'A' in "+str);
                    }
                    // If rX or rY have negative signs, these are dropped;
                    // the absolute value is used instead.
                    double rx = tt.nval;
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("ry coordinate missing for 'A' in "+str);
                    }
                    double ry = tt.nval;
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("x-axis-rotation missing for 'A' in "+str);
                    }
                    double xAxisRotation = tt.nval;
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("large-arc-flag missing for 'A' in "+str);
                    }
                    boolean largeArcFlag = tt.nval != 0;
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("sweep-flag missing for 'A' in "+str);
                    }
                    boolean sweepFlag = tt.nval != 0;
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("x coordinate missing for 'A' in "+str);
                    }
                    p.x += tt.nval;
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("y coordinate missing for 'A' in "+str);
                    }
                    p.y += tt.nval;
                    
                    path.arcTo(rx, ry, xAxisRotation, largeArcFlag, sweepFlag, p.x, p.y);
                    
                    nextCommand = 'a';
                    break;
                    }
                default :
                    System.out.println("SVGInputFormat.toPath aborting after illegal path command: "+command+" found in path "+str);
                    break Commands;
                    //throw new IOException("Illegal command: "+command);
            }
        }
        if (path != null) {
            paths.add(path);
        }
        return paths.toArray(new BezierPath[paths.size()]);
    }
    /* Reads core attributes as listed in
     * http://www.w3.org/TR/SVGMobile12/feature.html#CoreAttribute
     */
    private void readCoreAttributes(IXMLElement elem, HashMap<AttributeKey,Object> a)
    throws IOException {
        // read "id" or "xml:id"
        identifiedElements.put(elem.getAttribute("id"), elem);
        identifiedElements.put(elem.getAttribute("xml:id"), elem);
        
        // XXX - Add
        // xml:base
        // xml:lang
        // xml:space
        // class
        
    }
    /* Reads text attributes as listed in
     * http://www.w3.org/TR/SVGMobile12/feature.html#Text
     */
    private void readTextAttributes(IXMLElement elem, Map<AttributeKey,Object> a)
    throws IOException {
        Object value;
        
        //'text-anchor'
        //Value:  	start | middle | end | inherit
        //Initial:  	start
        //Applies to:  	'text' IXMLElement
        //Inherited:  	yes
        //Percentages:  	N/A
        //Media:  	visual
        //Animatable:  	yes
        //Computed value:  	 Specified value, except inherit
        value = readInheritAttribute(elem, "text-anchor", "start");
        // XXX - Implement me properly
        TEXT_ANCHOR.set(a, textAnchorMap.get(value));
        
        //'display-align'
        //Value:  	auto | before | center | after | inherit
        //Initial:  	auto
        //Applies to:  	'textArea'
        //Inherited:  	yes
        //Percentages:  	N/A
        //Media:  	visual
        //Animatable:  	yes
        //Computed value:  	 Specified value, except inherit
        value = readInheritAttribute(elem, "display-align", "auto");
        // XXX - Implement me properly
        if (! value.equals("auto")) {
            if (value.equals("center")) {
                TEXT_ANCHOR.set(a, TextAnchor.MIDDLE);
            } else if (value.equals("before")) {
                TEXT_ANCHOR.set(a, TextAnchor.END);
            }
        }
        
        //text-align
        //Value:	 start | end | center | inherit
        //Initial:	 start
        //Applies to:	 textArea elements
        //Inherited:	 yes
        //Percentages:	 N/A
        //Media:	 visual
        //Animatable:	 yes
        value = readInheritAttribute(elem, "text-align", "start");
        // XXX - Implement me properly
        if (! value.equals("start")) {
            TEXT_ANCHOR.set(a, textAnchorMap.get(value));
        }
    }
    /* Reads text flow attributes as listed in
     * http://www.w3.org/TR/SVGMobile12/feature.html#TextFlow
     */
    private void readTextFlowAttributes(IXMLElement elem, HashMap<AttributeKey,Object> a)
    throws IOException {
        Object value;
        
        //'line-increment'
        //Value:  	auto | <number> | inherit
        //Initial:  	auto
        //Applies to:  	'textArea'
        //Inherited:  	yes
        //Percentages:  	N/A
        //Media:  	visual
        //Animatable:  	yes
        //Computed value:  	 Specified value, except inherit
        value = readInheritAttribute(elem, "line-increment", "auto");
        System.out.println("SVGInputFormat not implemented line-increment="+value);
        
    }
    /* Reads the transform attribute as specified in
     * http://www.w3.org/TR/SVGMobile12/coords.html#TransformAttribute
     */
    private void readTransformAttribute(IXMLElement elem, HashMap<AttributeKey,Object> a)
    throws IOException {
        String value;
        value = readAttribute(elem, "transform", "none");
        if (! value.equals("none")) {
            TRANSFORM.set(a, toTransform(elem, value));
        }
    }
    /* Reads solid color attributes.
     */
    private void readSolidColorElement(IXMLElement elem)
    throws IOException {
        HashMap<AttributeKey,Object> a = new HashMap<AttributeKey,Object>();
        readCoreAttributes(elem, a);
        
        // 'solid-color'
        //Value:	 currentColor | <color> | inherit
        //Initial:	 black
        //Applies to:	 'solidColor' elements
        //Inherited:	 no
        //Percentages:	 N/A
        //Media:	 visual
        //Animatable:	 yes
        //Computed value:  	 Specified <color> value, except inherit
        Color color = toColor(elem, readAttribute(elem, "solid-color", "black"));
        
        //'solid-opacity'
        //Value:	<opacity-value> | inherit
        //Initial:	 1
        //Applies to:	 'solidColor' elements
        //Inherited:	 no
        //Percentages:	 N/A
        //Media:	 visual
        //Animatable:	 yes
        //Computed value:  	 Specified value, except inherit
        double opacity = toDouble(elem, readAttribute(elem, "solid-opacity", "1"), 1, 0, 1);
        if (opacity != 1) {
            color = new Color(((int) (255 * opacity) << 24) | (0xffffff & color.getRGB()), true);
        }
        
        elementObjects.put(elem, color);
        
        
    }
    /** Reads image attributes.
     */
    private void readImageAttributes(IXMLElement elem, HashMap<AttributeKey,Object> a)
    throws IOException {
        double value;
        
        //'opacity'
        //Value:  	<opacity-value> | inherit
        //Initial:  	1
        //Applies to:  	 'image' element
        //Inherited:  	no
        //Percentages:  	N/A
        //Media:  	visual
        //Animatable:  	yes
        //Computed value:  	 Specified value, except inherit
        value = toDouble(elem, readAttribute(elem, "opacity", "1"), 1, 0, 1);
        if (value != 1) {
            System.out.println("SVGInputFormat not implemented  opacity="+value);
        }
        
    }
    /** Reads shape attributes.
     */
    private void readShapeAttributes(IXMLElement elem, HashMap<AttributeKey,Object> a)
    throws IOException {
        Object objectValue;
        String value;
        double doubleValue;
        
        //'color'
        // Value:  	<color> | inherit
        // Initial:  	 depends on user agent
        // Applies to:  	None. Indirectly affects other properties via currentColor
        // Inherited:  	 yes
        // Percentages:  	 N/A
        // Media:  	 visual
        // Animatable:  	 yes
        // Computed value:  	 Specified <color> value, except inherit
        //
        // value = readInheritAttribute(elem, "color", "black");
        // System.out.println("color="+value);
        
        //'color-rendering'
        // Value:  	 auto | optimizeSpeed | optimizeQuality | inherit
        // Initial:  	 auto
        // Applies to:  	 container elements , graphics elements and 'animateColor'
        // Inherited:  	 yes
        // Percentages:  	 N/A
        // Media:  	 visual
        // Animatable:  	 yes
        // Computed value:  	 Specified value, except inherit
        //
        // value = readInheritAttribute(elem, "color-rendering", "auto");
        // System.out.println("color-rendering="+value);
        
        // 'fill'
        // Value:  	<paint> | inherit (See Specifying paint)
        // Initial:  	 black
        // Applies to:  	 shapes and text content elements
        // Inherited:  	 yes
        // Percentages:  	 N/A
        // Media:  	 visual
        // Animatable:  	 yes
        // Computed value:  	 "none", system paint, specified <color> value or absolute IRI
        objectValue = toPaint(elem, readInheritAttribute(elem, "fill", "black"));
        if (objectValue instanceof Color) {
            FILL_COLOR.set(a, (Color) objectValue);
        } else if (objectValue instanceof Gradient) {
            FILL_GRADIENT.set(a, (Gradient) objectValue);
        } else if (objectValue == null) {
            FILL_COLOR.set(a, null);
        } else {
            FILL_COLOR.set(a, null);
            System.out.println("SVGInputFormat not implemented  fill="+objectValue);
        }
        
        //'fill-opacity'
        //Value:  	 <opacity-value> | inherit
        //Initial:  	 1
        //Applies to:  	 shapes and text content elements
        //Inherited:  	 yes
        //Percentages:  	 N/A
        //Media:  	 visual
        //Animatable:  	 yes
        //Computed value:  	 Specified value, except inherit
        objectValue = readInheritAttribute(elem, "fill-opacity", null);
        if (objectValue == null) {
            objectValue = readInheritAttribute(elem, "opacity", "1");
        }
        FILL_OPACITY.set(a, toDouble(elem, (String) objectValue, 1d, 0d, 1d));
        
        // 'fill-rule'
        // Value:	 nonzero | evenodd | inherit
        // Initial: 	 nonzero
        // Applies to:  	 shapes and text content elements
        // Inherited:  	 yes
        // Percentages:  	 N/A
        // Media:  	 visual
        // Animatable:  	 yes
        // Computed value:  	 Specified value, except inherit
        value = readInheritAttribute(elem, "fill-rule", "nonzero");
        WINDING_RULE.set(a, fillRuleMap.get(value));
        
        //'stroke'
        //Value:  	<paint> | inherit (See Specifying paint)
        //Initial:  	 none
        //Applies to:  	 shapes and text content elements
        //Inherited:  	 yes
        //Percentages:  	 N/A
        //Media:  	 visual
        //Animatable:  	 yes
        //Computed value:  	 "none", system paint, specified <color> value
        // or absolute IRI
        objectValue = toPaint(elem, readInheritAttribute(elem, "stroke", "none"));
        if (objectValue instanceof Color) {
            STROKE_COLOR.set(a, (Color) objectValue);
        } else if (objectValue instanceof Gradient) {
            STROKE_GRADIENT.set(a, (Gradient) objectValue);
        } else if (objectValue == null) {
            STROKE_COLOR.set(a, null);
        } else {
            STROKE_COLOR.set(a, null);
            System.out.println("SVGInputFormat not implemented  stroke="+objectValue);
        }
        
        //'stroke-dasharray'
        //Value:  	 none | <dasharray> | inherit
        //Initial:  	 none
        //Applies to:  	 shapes and text content elements
        //Inherited:  	 yes
        //Percentages:  	 N/A
        //Media:  	 visual
        //Animatable:  	 yes (non-additive)
        //Computed value:  	 Specified value, except inherit
        value = readInheritAttribute(elem, "stroke-dasharray", "none");
        if (! value.equals("none")) {
            String[] values = toCommaSeparatedArray(value);
            double[] dashes = new double[values.length];
            for (int i=0; i < values.length; i++) {
                dashes[i] = toNumber(elem, values[i]);
            }
            STROKE_DASHES.set(a, dashes);
        }
        
        //'stroke-dashoffset'
        //Value:  	<length> | inherit
        //Initial:  	 0
        //Applies to:  	 shapes and text content elements
        //Inherited:  	 yes
        //Percentages:  	 N/A
        //Media:  	 visual
        //Animatable:  	 yes
        //Computed value:  	 Specified value, except inherit
        doubleValue = toNumber(elem, readInheritAttribute(elem, "stroke-dashoffset", "0"));
        STROKE_DASH_PHASE.set(a, doubleValue);
        IS_STROKE_DASH_FACTOR.set(a, false);
        
        //'stroke-linecap'
        //Value:  	 butt | round | square | inherit
        //Initial:  	 butt
        //Applies to:  	 shapes and text content elements
        //Inherited:  	 yes
        //Percentages:  	 N/A
        //Media:  	 visual
        //Animatable:  	 yes
        //Computed value:  	 Specified value, except inherit
        value = readInheritAttribute(elem, "stroke-linecap", "butt");
        STROKE_CAP.set(a, strokeLinecapMap.get(value));
        
        
        //'stroke-linejoin'
        //Value:  	 miter | round | bevel | inherit
        //Initial:  	 miter
        //Applies to:  	 shapes and text content elements
        //Inherited:  	 yes
        //Percentages:  	 N/A
        //Media:  	 visual
        //Animatable:  	 yes
        //Computed value:  	 Specified value, except inherit
        value = readInheritAttribute(elem, "stroke-linejoin", "miter");
        STROKE_JOIN.set(a, strokeLinejoinMap.get(value));
        
        //'stroke-miterlimit'
        //Value:  	 <miterlimit> | inherit
        //Initial:  	 4
        //Applies to:  	 shapes and text content elements
        //Inherited:  	 yes
        //Percentages:  	 N/A
        //Media:  	 visual
        //Animatable:  	 yes
        //Computed value:  	 Specified value, except inherit
        doubleValue = toDouble(elem, readInheritAttribute(elem, "stroke-miterlimit", "4"), 4d, 1d, Double.MAX_VALUE);
        STROKE_MITER_LIMIT.set(a, doubleValue);
        IS_STROKE_MITER_LIMIT_FACTOR.set(a, false);
        
        //'stroke-opacity'
        //Value:  	 <opacity-value> | inherit
        //Initial:  	 1
        //Applies to:  	 shapes and text content elements
        //Inherited:  	 yes
        //Percentages:  	 N/A
        //Media:  	 visual
        //Animatable:  	 yes
        //Computed value:  	 Specified value, except inherit
        objectValue = readInheritAttribute(elem, "stroke-opacity", null);
        if (objectValue == null) {
            objectValue = readInheritAttribute(elem, "opacity", "1");
        }
        STROKE_OPACITY.set(a, toDouble(elem, (String) objectValue, 1d, 0d, 1d));
        
        //'stroke-width'
        //Value:  	<length> | inherit
        //Initial:  	 1
        //Applies to:  	 shapes and text content elements
        //Inherited:  	 yes
        //Percentages:  	 N/A
        //Media:  	 visual
        //Animatable:  	 yes
        //Computed value:  	 Specified value, except inherit
        doubleValue = toScaledNumber(elem, readInheritAttribute(elem, "stroke-width", "1"));
        STROKE_WIDTH.set(a, doubleValue);
    }
    /* Reads shape attributes for the SVG "use" element.
     */
    private void readUseShapeAttributes(IXMLElement elem, HashMap<AttributeKey,Object> a)
    throws IOException {
        Object objectValue;
        String value;
        double doubleValue;
        
        //'color'
        // Value:  	<color> | inherit
        // Initial:  	 depends on user agent
        // Applies to:  	None. Indirectly affects other properties via currentColor
        // Inherited:  	 yes
        // Percentages:  	 N/A
        // Media:  	 visual
        // Animatable:  	 yes
        // Computed value:  	 Specified <color> value, except inherit
        //
        // value = readInheritAttribute(elem, "color", "black");
        // System.out.println("color="+value);
        
        //'color-rendering'
        // Value:  	 auto | optimizeSpeed | optimizeQuality | inherit
        // Initial:  	 auto
        // Applies to:  	 container elements , graphics elements and 'animateColor'
        // Inherited:  	 yes
        // Percentages:  	 N/A
        // Media:  	 visual
        // Animatable:  	 yes
        // Computed value:  	 Specified value, except inherit
        //
        // value = readInheritAttribute(elem, "color-rendering", "auto");
        // System.out.println("color-rendering="+value);
        
        // 'fill'
        // Value:  	<paint> | inherit (See Specifying paint)
        // Initial:  	 black
        // Applies to:  	 shapes and text content elements
        // Inherited:  	 yes
        // Percentages:  	 N/A
        // Media:  	 visual
        // Animatable:  	 yes
        // Computed value:  	 "none", system paint, specified <color> value or absolute IRI
        objectValue = readInheritAttribute(elem, "fill", null);
        if (objectValue != null) {
            objectValue = toPaint(elem, (String) objectValue);
            if (objectValue instanceof Color) {
                FILL_COLOR.set(a, (Color) objectValue);
            } else if (objectValue instanceof Gradient) {
                FILL_GRADIENT.set(a, (Gradient) objectValue);
            } else if (objectValue == null) {
                FILL_COLOR.set(a, null);
            } else {
                FILL_COLOR.set(a, null);
                System.out.println("SVGInputFormat not implemented  fill="+objectValue);
            }
        }
        
        //'fill-opacity'
        //Value:  	 <opacity-value> | inherit
        //Initial:  	 1
        //Applies to:  	 shapes and text content elements
        //Inherited:  	 yes
        //Percentages:  	 N/A
        //Media:  	 visual
        //Animatable:  	 yes
        //Computed value:  	 Specified value, except inherit
        objectValue = readInheritAttribute(elem, "fill-opacity", null);
        if (objectValue == null) {
            objectValue = readInheritAttribute(elem, "opacity", null);
        }
        if (objectValue != null) {
            FILL_OPACITY.set(a, toDouble(elem, (String) objectValue, 1d, 0d, 1d));
        }
        
        // 'fill-rule'
        // Value:	 nonzero | evenodd | inherit
        // Initial: 	 nonzero
        // Applies to:  	 shapes and text content elements
        // Inherited:  	 yes
        // Percentages:  	 N/A
        // Media:  	 visual
        // Animatable:  	 yes
        // Computed value:  	 Specified value, except inherit
        value = readInheritAttribute(elem, "fill-rule", null);
        if (value != null) {
            WINDING_RULE.set(a, fillRuleMap.get(value));
        }
        
        //'stroke'
        //Value:  	<paint> | inherit (See Specifying paint)
        //Initial:  	 none
        //Applies to:  	 shapes and text content elements
        //Inherited:  	 yes
        //Percentages:  	 N/A
        //Media:  	 visual
        //Animatable:  	 yes
        //Computed value:  	 "none", system paint, specified <color> value
        // or absolute IRI
        objectValue = toPaint(elem, readInheritAttribute(elem, "stroke", null));
        if (objectValue != null) {
            if (objectValue instanceof Color) {
                STROKE_COLOR.set(a, (Color) objectValue);
            } else if (objectValue instanceof Gradient) {
                STROKE_GRADIENT.set(a, (Gradient) objectValue);
            }
        }
        
        //'stroke-dasharray'
        //Value:  	 none | <dasharray> | inherit
        //Initial:  	 none
        //Applies to:  	 shapes and text content elements
        //Inherited:  	 yes
        //Percentages:  	 N/A
        //Media:  	 visual
        //Animatable:  	 yes (non-additive)
        //Computed value:  	 Specified value, except inherit
        value = readInheritAttribute(elem, "stroke-dasharray", null);
        if (value != null && ! value.equals("none")) {
            String[] values = toCommaSeparatedArray(value);
            double[] dashes = new double[values.length];
            for (int i=0; i < values.length; i++) {
                dashes[i] = toNumber(elem, values[i]);
            }
            STROKE_DASHES.set(a, dashes);
        }
        
        //'stroke-dashoffset'
        //Value:  	<length> | inherit
        //Initial:  	 0
        //Applies to:  	 shapes and text content elements
        //Inherited:  	 yes
        //Percentages:  	 N/A
        //Media:  	 visual
        //Animatable:  	 yes
        //Computed value:  	 Specified value, except inherit
        objectValue = readInheritAttribute(elem, "stroke-dashoffset", null);
        if (objectValue != null) {
            doubleValue = toNumber(elem, (String) objectValue);
            STROKE_DASH_PHASE.set(a, doubleValue);
            IS_STROKE_DASH_FACTOR.set(a, false);
        }
        
        //'stroke-linecap'
        //Value:  	 butt | round | square | inherit
        //Initial:  	 butt
        //Applies to:  	 shapes and text content elements
        //Inherited:  	 yes
        //Percentages:  	 N/A
        //Media:  	 visual
        //Animatable:  	 yes
        //Computed value:  	 Specified value, except inherit
        value = readInheritAttribute(elem, "stroke-linecap", null);
        if (value != null) {
            STROKE_CAP.set(a, strokeLinecapMap.get(value));
        }
        
        //'stroke-linejoin'
        //Value:  	 miter | round | bevel | inherit
        //Initial:  	 miter
        //Applies to:  	 shapes and text content elements
        //Inherited:  	 yes
        //Percentages:  	 N/A
        //Media:  	 visual
        //Animatable:  	 yes
        //Computed value:  	 Specified value, except inherit
        value = readInheritAttribute(elem, "stroke-linejoin", null);
        if (value != null) {
            STROKE_JOIN.set(a, strokeLinejoinMap.get(value));
        }
        //'stroke-miterlimit'
        //Value:  	 <miterlimit> | inherit
        //Initial:  	 4
        //Applies to:  	 shapes and text content elements
        //Inherited:  	 yes
        //Percentages:  	 N/A
        //Media:  	 visual
        //Animatable:  	 yes
        //Computed value:  	 Specified value, except inherit
        objectValue = readInheritAttribute(elem, "stroke-miterlimit", null);
        if (objectValue != null) {
            doubleValue = toDouble(elem, (String) objectValue, 4d, 1d, Double.MAX_VALUE);
            STROKE_MITER_LIMIT.set(a, doubleValue);
            IS_STROKE_MITER_LIMIT_FACTOR.set(a, false);
        }
        
        //'stroke-opacity'
        //Value:  	 <opacity-value> | inherit
        //Initial:  	 1
        //Applies to:  	 shapes and text content elements
        //Inherited:  	 yes
        //Percentages:  	 N/A
        //Media:  	 visual
        //Animatable:  	 yes
        //Computed value:  	 Specified value, except inherit
        objectValue = readInheritAttribute(elem, "stroke-opacity", null);
        if (objectValue == null) {
            objectValue = readInheritAttribute(elem, "opacity", null);
        }
        if (objectValue != null) {
            STROKE_OPACITY.set(a, toDouble(elem, (String) objectValue, 1d, 0d, 1d));
        }
        
        //'stroke-width'
        //Value:  	<length> | inherit
        //Initial:  	 1
        //Applies to:  	 shapes and text content elements
        //Inherited:  	 yes
        //Percentages:  	 N/A
        //Media:  	 visual
        //Animatable:  	 yes
        //Computed value:  	 Specified value, except inherit
        objectValue = readInheritAttribute(elem, "stroke-width", null);
        if (objectValue != null) {
            doubleValue = toScaledNumber(elem, (String) objectValue);
            STROKE_WIDTH.set(a, doubleValue);
        }
    }
    /* Reads viewport attributes.
     */
    private void readViewportAttributes(IXMLElement elem, HashMap<AttributeKey,Object> a)
    throws IOException {
        Object value;
        //'viewport-fill'
        //Value:	 "none" | <color> | inherit
        //Initial:	 none
        //Applies to:	viewport-creating elements
        //Inherited:	 no
        //Percentages:	 N/A
        //Media:	 visual
        //Animatable:	 yes
        //Computed value:  	 "none" or specified <color> value, except inherit
        value = readAttribute(elem, "viewport-fill", "none");
        System.out.println("SVGInputFormat not implemented  viewport-fill="+value);
        
        //'viewport-fill-opacity'
        //Value:	<opacity-value> | inherit
        //Initial:	 1.0
        //Applies to:	viewport-creating elements
        //Inherited:	 no
        //Percentages:	 N/A
        //Media:	 visual
        //Animatable:	 yes
        //Computed value:  	 Specified value, except inherit
        value = readAttribute(elem, "viewport-fill-opacity", "1.0");
        System.out.println("SVGInputFormat not implemented  viewport-fill-opacity="+value);
        
    }
    /* Reads graphics attributes as listed in
     * http://www.w3.org/TR/SVGMobile12/feature.html#GraphicsAttribute
     */
    private void readGraphicsAttributes(IXMLElement elem, Figure f)
    throws IOException {
        Object value;
        // 'display'
        // Value:  	 inline | block | list-item |
        // run-in | compact | marker |
        // table | inline-table | table-row-group | table-header-group |
        // table-footer-group | table-row | table-column-group | table-column |
        // table-cell | table-caption | none | inherit
        // Initial:  	 inline
        // Applies to:  	 'svg' , 'g' , 'switch' , 'a' , 'foreignObject' ,
        // graphics elements (including the text content block elements) and text
        // sub-elements (for example, 'tspan' and 'a' )
        // Inherited:  	 no
        // Percentages:  	 N/A
        // Media:  	 all
        // Animatable:  	 yes
        // Computed value:  	 Specified value, except inherit
        value = readAttribute(elem, "display", "inline");
        System.out.println("SVGInputFormat not implemented display="+value);
        
        
        //'image-rendering'
        //Value:  	 auto | optimizeSpeed | optimizeQuality | inherit
        //Initial:  	 auto
        //Applies to:  	 images
        //Inherited:  	 yes
        //Percentages:  	 N/A
        //Media:  	 visual
        //Animatable:  	 yes
        //Computed value:  	 Specified value, except inherit
        value = readInheritAttribute(elem, "image-rendering", "auto");
        System.out.println("SVGInputFormat not implemented image-rendering="+value);
        
        //'pointer-events'
        //Value:  	boundingBox | visiblePainted | visibleFill | visibleStroke | visible |
        //painted | fill | stroke | all | none | inherit
        //Initial:  	visiblePainted
        //Applies to:  	graphics elements
        //Inherited:  	yes
        //Percentages:  	N/A
        //Media:  	visual
        //Animatable:  	yes
        //Computed value:  	Specified value, except inherit
        value = readInheritAttribute(elem, "pointer-events", "visiblePainted");
        System.out.println("SVGInputFormat not implemented pointer-events="+value);
        
        // 'shape-rendering'
        //Value:  	 auto | optimizeSpeed | crispEdges |
        //geometricPrecision | inherit
        //Initial:  	 auto
        //Applies to:  	 shapes
        //Inherited:  	 yes
        //Percentages:  	 N/A
        //Media:  	 visual
        //Animatable:  	 yes
        //Computed value:  	 Specified value, except inherit
        value = readInheritAttribute(elem, "shape-rendering", "auto");
        System.out.println("SVGInputFormat not implemented shape-rendering="+value);
        
        //'text-rendering'
        //Value:  	 auto | optimizeSpeed | optimizeLegibility |
        //geometricPrecision | inherit
        //Initial:  	 auto
        //Applies to:  	text content block elements
        //Inherited:  	 yes
        //Percentages:  	 N/A
        //Media:  	 visual
        //Animatable:  	 yes
        //Computed value:  	 Specified value, except inherit
        value = readInheritAttribute(elem, "text-rendering", "auto");
        System.out.println("SVGInputFormat not implemented text-rendering="+value);
        
        //'vector-effect'
        //Value:  	 non-scaling-stroke | none | inherit
        //Initial:  	 none
        //Applies to:  	 graphics elements
        //Inherited:  	 no
        //Percentages:  	 N/A
        //Media:  	 visual
        //Animatable:  	 yes
        //Computed value:  	 Specified value, except inherit
        value = readAttribute(elem, "vector-effect", "none");
        System.out.println("SVGInputFormat not implemented vector-effect="+value);
        
        //'visibility'
        //Value:  	 visible | hidden | collapse | inherit
        //Initial:  	 visible
        //Applies to:  	 graphics elements (including the text content block
        // elements) and text sub-elements (for example, 'tspan' and 'a' )
        //Inherited:  	 yes
        //Percentages:  	 N/A
        //Media:  	 visual
        //Animatable:  	 yes
        //Computed value:  	 Specified value, except inherit
        value = readInheritAttribute(elem, "visibility", null);
        System.out.println("SVGInputFormat not implemented visibility="+value);
    }
    /**
     * Reads an SVG "linearGradient" element.
     */
    private void readLinearGradientElement(IXMLElement elem)
    throws IOException {
        HashMap<AttributeKey,Object> a = new HashMap<AttributeKey,Object>();
        readCoreAttributes(elem, a);
        
        double x1 = toNumber(elem, readAttribute(elem, "x1", "0"));
        double y1 = toNumber(elem, readAttribute(elem, "y1", "0"));
        double x2 = toNumber(elem, readAttribute(elem, "x2", "1"));
        double y2 = toNumber(elem, readAttribute(elem, "y2", "0"));
        boolean isRelativeToFigureBounds = readAttribute(elem, "gradientUnits", "objectBoundingBox").equals("objectBoundingBox");
        
        ArrayList<IXMLElement> stops = elem.getChildrenNamed("stop",SVG_NAMESPACE);
        if (stops.size() == 0) {
            stops = elem.getChildrenNamed("stop");
        }
        if (stops.size() == 0) {
            // FIXME - Implement xlink support throughouth SVGInputFormat
            String xlink = readAttribute(elem, "xlink:href", "");
            if (xlink.startsWith("#") &&
                    elementObjects.get(identifiedElements.get(xlink.substring(1))) != null) {
                
                stops = identifiedElements.get(xlink.substring(1)).getChildrenNamed("stop",SVG_NAMESPACE);
                if (stops.size() == 0) {
                    stops = identifiedElements.get(xlink.substring(1)).getChildrenNamed("stop");
                }
            }
        }
        if (stops.size() == 0) {
            System.out.println("SVGInpuFormat: Warning no stops in linearGradient "+elem);
        }
        
        AffineTransform tx = toTransform(elem, readAttribute(elem, "gradientTransform", "none"));
        if (tx != null) {
            Point2D.Double p = new Point2D.Double(x1,y1);
            tx.transform(p,p);
            x1 = p.x;
            y1 = p.y;
            p.x = x2;
            p.y = y2;
            tx.transform(p,p);
            x2 = p.x;
            y2 = p.y;
        }
        
        double[] stopOffsets = new double[stops.size()];
        Color[] stopColors = new Color[stops.size()];
        for (int i=0; i < stops.size(); i++) {
            IXMLElement stopElem = stops.get(i);
            String offsetStr = readAttribute(stopElem, "offset", "0");
            if (offsetStr.endsWith("%")) {
                stopOffsets[i] = toDouble(stopElem, offsetStr.substring(0, offsetStr.length() - 1), 0, 0, 100) / 100d;
            } else {
                stopOffsets[i] = toDouble(stopElem, offsetStr, 0, 0, 1);
            }
            // 'stop-color'
            // Value:  	currentColor | <color> | inherit
            // Initial:  	black
            // Applies to:  	 'stop' elements
            // Inherited:  	no
            // Percentages:  	N/A
            // Media:  	visual
            // Animatable:  	yes
            // Computed value:  	 Specified <color> value, except i
            stopColors[i] = toColor(stopElem, readAttribute(stopElem, "stop-color", "black"));
            //'stop-opacity'
            //Value:  	<opacity-value> | inherit
            //Initial:  	1
            //Applies to:  	 'stop' elements
            //Inherited:  	no
            //Percentages:  	N/A
            //Media:  	visual
            //Animatable:  	yes
            //Computed value:  	 Specified value, except inherit
            double doubleValue = toDouble(stopElem, readAttribute(stopElem, "stop-opacity", "1"), 1, 0, 1);
            if (doubleValue != 1) {
                stopColors[i] = new Color(((int) (doubleValue * 255) << 24) | (stopColors[i].getRGB() & 0xffffff), true);
            }
        }
        
        Gradient gradient = factory.createLinearGradient(
                x1, y1, x2, y2,
                stopOffsets, stopColors,
                isRelativeToFigureBounds
                );
        elementObjects.put(elem, gradient);
    }
    /**
     * Reads an SVG "radialGradient" element.
     */
    private void readRadialGradientElement(IXMLElement elem)
    throws IOException {
        HashMap<AttributeKey,Object> a = new HashMap<AttributeKey,Object>();
        readCoreAttributes(elem, a);
        
        // XXX - Implement me
        double cx = toNumber(elem, readAttribute(elem, "cx", "0"));
        double cy = toNumber(elem, readAttribute(elem, "cy", "0"));
        double r = toNumber(elem, readAttribute(elem, "r", "0.5"));
        boolean isRelativeToFigureBounds = readAttribute(elem, "gradientUnits", "objectBoundingBox").equals("objectBoundingBox");
        
        ArrayList<IXMLElement> stops = elem.getChildrenNamed("stop",SVG_NAMESPACE);
        if (stops.size() == 0) {
            stops = elem.getChildrenNamed("stop");
        }
        if (stops.size() == 0) {
            // FIXME - Implement xlink support throughout SVGInputFormat
            String xlink = readAttribute(elem, "xlink:href", "");
            if (xlink.startsWith("#") &&
                    elementObjects.get(identifiedElements.get(xlink.substring(1))) != null) {
                
                stops = identifiedElements.get(xlink.substring(1)).getChildrenNamed("stop",SVG_NAMESPACE);
                if (stops.size() == 0) {
                    stops = identifiedElements.get(xlink.substring(1)).getChildrenNamed("stop");
                }
            }
        }
        
        double[] stopOffsets = new double[stops.size()];
        Color[] stopColors = new Color[stops.size()];
        for (int i=0; i < stops.size(); i++) {
            IXMLElement stopElem = stops.get(i);
            String offsetStr = readAttribute(stopElem, "offset", "0");
            if (offsetStr.endsWith("%")) {
                stopOffsets[i] = toDouble(stopElem, offsetStr.substring(0, offsetStr.length() - 1), 0, 0, 100) / 100d;
            } else {
                stopOffsets[i] = toDouble(stopElem, offsetStr, 0, 0, 1);
            }
            // 'stop-color'
            // Value:  	currentColor | <color> | inherit
            // Initial:  	black
            // Applies to:  	 'stop' elements
            // Inherited:  	no
            // Percentages:  	N/A
            // Media:  	visual
            // Animatable:  	yes
            // Computed value:  	 Specified <color> value, except i
            stopColors[i] = toColor(stopElem, readAttribute(stopElem, "stop-color", "black"));
            //'stop-opacity'
            //Value:  	<opacity-value> | inherit
            //Initial:  	1
            //Applies to:  	 'stop' elements
            //Inherited:  	no
            //Percentages:  	N/A
            //Media:  	visual
            //Animatable:  	yes
            //Computed value:  	 Specified value, except inherit
            double doubleValue = toDouble(stopElem, readAttribute(stopElem, "stop-opacity", "1"), 1, 0, 1);
            if (doubleValue != 1) {
                stopColors[i] = new Color(((int) (doubleValue * 255) << 24) | (stopColors[i].getRGB() & 0xffffff), true);
            }
        }
        
        Gradient gradient = factory.createRadialGradient(
                cx, cy, r,
                stopOffsets, stopColors,
                isRelativeToFigureBounds
                );
        elementObjects.put(elem, gradient);
    }
    /* Reads font attributes as listed in
     * http://www.w3.org/TR/SVGMobile12/feature.html#Font
     */
    private void readFontAttributes(IXMLElement elem, Map<AttributeKey,Object> a)
    throws IOException {
        String value;
        double doubleValue;
        
        // 'font-family'
        // Value:  	[[ <family-name> |
        // <generic-family> ],]* [<family-name> |
        // <generic-family>] | inherit
        // Initial:  	depends on user agent
        // Applies to:  	text content elements
        // Inherited:  	yes
        // Percentages:  	N/A
        // Media:  	visual
        // Animatable:  	yes
        // Computed value:  	 Specified value, except inherit
        value = readInheritAttribute(elem, "font-family", "Dialog");
        FONT_FACE.set(a, new Font(value, Font.PLAIN, 12));
        
        // 'font-size'
        // Value:  	<absolute-size> | <relative-size> |
        // <length> | inherit
        // Initial:  	medium
        // Applies to:  	text content elements
        // Inherited:  	yes, the computed value is inherited
        // Percentages:  	N/A
        // Media:  	visual
        // Animatable:  	yes
        // Computed value:  	 Absolute length
        doubleValue = readInheritFontSizeAttribute(elem, "font-size", "medium");
        FONT_SIZE.set(a, doubleValue);
        
        // 'font-style'
        // Value:  	normal | italic | oblique | inherit
        // Initial:  	normal
        // Applies to:  	text content elements
        // Inherited:  	yes
        // Percentages:  	N/A
        // Media:  	visual
        // Animatable:  	yes
        // Computed value:  	 Specified value, except inherit
        value = readInheritAttribute(elem, "font-style", "normal");
        FONT_ITALIC.set(a, value.equals("italic"));
        
        
        //'font-variant'
        //Value:  	normal | small-caps | inherit
        //Initial:  	normal
        //Applies to:  	text content elements
        //Inherited:  	yes
        //Percentages:  	N/A
        //Media:  	visual
        //Animatable:  	no
        //Computed value:  	 Specified value, except inherit
        value = readInheritAttribute(elem, "font-variant", "normal");
        // System.out.println("font-variant="+value);
        
        // 'font-weight'
        // Value:  	normal | bold | bolder | lighter | 100 | 200 | 300
        // | 400 | 500 | 600 | 700 | 800 | 900 | inherit
        // Initial:  	normal
        // Applies to:  	text content elements
        // Inherited:  	yes
        // Percentages:  	N/A
        // Media:  	visual
        // Animatable:  	yes
        // Computed value:  	 one of the legal numeric values, non-numeric
        // values shall be converted to numeric values according to the rules
        // defined below.
        value = readInheritAttribute(elem, "font-weight", "normal");
        FONT_BOLD.set(a, value.equals("bold") || value.equals("bolder") ||
                value.equals("400") || value.equals("500") || value.equals("600") ||
                value.equals("700") || value.equals("800") || value.equals("900"));
        
    }
    
    /**
     * Reads a paint style attribute. This can be a Color or a Gradient or null.
     * XXX - Doesn't support url(...) colors yet.
     */
    private Object toPaint(IXMLElement elem, String value) throws IOException {
        String str = value;
        if (str == null) {
            return null;
            /*
            if (elem.getParent() != null && elem.getParent().getNamespace().equals(SVG_NAMESPACE)) {
                return readPaint(elem.getParent(), attributeName);
            }*/
        }
        
        
        str = str.trim().toLowerCase();
        
        if (str.equals("none")) {
            return null;
        } else if (str.equals("currentColor")) {
            // XXX - This may cause endless recursion
            return toPaint(elem, readAttribute(elem, "color", "black"));
        } else if (SVG_COLORS.containsKey(str)) {
            return SVG_COLORS.get(str);
        } else if (str.startsWith("#") && str.length() == 7) {
            return new Color(Integer.decode(str));
        } else if (str.startsWith("#") && str.length() == 4) {
            // Three digits hex value
            int th = Integer.decode(str);
            return new Color(
                    (th & 0xf) | ((th & 0xf) << 4) |
                    ((th & 0xf0) << 4) | ((th & 0xf0) << 8) |
                    ((th & 0xf00) << 8) | ((th & 0xf00) << 12)
                    );
        } else if (str.startsWith("rgb")) {
            StringTokenizer tt = new StringTokenizer(str,"() ,");
            tt.nextToken();
            Color c = new Color(
                    Integer.decode(tt.nextToken()),
                    Integer.decode(tt.nextToken()),
                    Integer.decode(tt.nextToken())
                    );
            return c;
        } else if (str.startsWith("url(")) {
            String href = value.substring(4,value.length() - 1);
            if (identifiedElements.containsKey(href.substring(1))) {
                Object obj = elementObjects.get(identifiedElements.get(href.substring(1)));
                return obj;
            }
            // XXX - Implement me
            System.out.println("SVGInputFormat.toPaint not implemented for "+href);
            return null;
        } else {
            return null;
        }
    }
    /**
     * Reads a color style attribute. This can be a Color or null.
     * XXX - Doesn't support url(...) colors yet.
     */
    private Color toColor(IXMLElement elem, String value) throws IOException {
        String str = value;
        if (str == null) {
            return null;
            /*
            if (elem.getParent() != null && elem.getParent().getNamespace().equals(SVG_NAMESPACE)) {
                return readPaint(elem.getParent(), attributeName);
            }*/
        }
        
        str = str.trim().toLowerCase();
        
        if (str.equals("currentColor")) {
            // XXX - This may cause endless recursion
            return toColor(elem, readAttribute(elem, "color", "black"));
        } else if (SVG_COLORS.containsKey(str)) {
            return SVG_COLORS.get(str);
        } else if (str.startsWith("#") && str.length() == 7) {
            return new Color(Integer.decode(str));
        } else if (str.startsWith("#") && str.length() == 4) {
            // Three digits hex value
            int th = Integer.decode(str);
            return new Color(
                    (th & 0xf) | ((th & 0xf) << 4) |
                    ((th & 0xf0) << 4) | ((th & 0xf0) << 8) |
                    ((th & 0xf00) << 8) | ((th & 0xf00) << 12)
                    );
        } else if (str.startsWith("rgb")) {
            StringTokenizer tt = new StringTokenizer(str,"() ,");
            tt.nextToken();
            Color c = new Color(
                    Integer.decode(tt.nextToken()),
                    Integer.decode(tt.nextToken()),
                    Integer.decode(tt.nextToken())
                    );
            return c;
        } else if (str.startsWith("url")) {
            // XXX - Implement me
            System.out.println("SVGInputFormat.toColor not implemented for "+str);
            return null;
        } else {
            return null;
        }
    }
    /**
     * Reads a double attribute.
     */
    private double toDouble(IXMLElement elem, String value) throws IOException {
        return toDouble(elem, value, 0, Double.MIN_VALUE, Double.MAX_VALUE);
    }
    /**
     * Reads a double attribute.
     */
    private double toDouble(IXMLElement elem, String value, double defaultValue, double min, double max) throws IOException {
        try {
            double d = Double.valueOf(value);
            return Math.max(Math.min(d, max), min);
        } catch (NumberFormatException e) {
            return defaultValue;
           /*
           IOException ex = new IOException(elem.getTagName()+"@"+elem.getLineNr()+" "+e.getMessage());
           ex.initCause(e);
           throw ex;*/
        }
    }
    /**
     * Reads a text attribute.
     * This method takes the "xml:space" attribute into account.
     * http://www.w3.org/TR/SVGMobile12/text.html#WhiteSpace
     */
    private String toText(IXMLElement elem, String value) throws IOException {
        String space = readInheritAttribute(elem,"xml:space","default");
        if (space.equals("default")) {
            return value.trim().replaceAll("\\s++"," ");
        } else /*if (space.equals("preserve"))*/ {
            return value;
        }
    }
    /* Converts an SVG transform attribute value into an AffineTransform
     * as specified in
     * http://www.w3.org/TR/SVGMobile12/coords.html#TransformAttribute
     */
    public static AffineTransform toTransform(IXMLElement elem, String str) throws IOException {
        AffineTransform t = new AffineTransform();
        
        if (str != null && ! str.equals("none")) {
            
            StreamTokenizer tt = new StreamTokenizer(new StringReader(str));
            tt.resetSyntax();
            tt.wordChars('a', 'z');
            tt.wordChars('A', 'Z');
            tt.wordChars(128 + 32, 255);
            tt.whitespaceChars(0, ' ');
            tt.whitespaceChars(',', ',');
            tt.parseNumbers();
            
            while (tt.nextToken() != StreamTokenizer.TT_EOF) {
                if (tt.ttype != StreamTokenizer.TT_WORD) {
                    throw new IOException("Illegal transform "+str);
                }
                String type = tt.sval;
                if (tt.nextToken() != '(') {
                    throw new IOException("'(' not found in transform "+str);
                }
                if (type.equals("matrix")) {
                    double[] m = new double[6];
                    for (int i=0; i < 6; i++) {
                        if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                            throw new IOException("Matrix value "+i+" not found in transform "+str+" token:"+tt.ttype+" "+tt.sval);
                        }
                        m[i] = tt.nval;
                        if (tt.nextToken() == StreamTokenizer.TT_WORD &&
                                (tt.sval.startsWith("E") || tt.sval.startsWith("e"))) {
                            double mantissa = tt.nval;
                            m[i] = Double.valueOf(m[i] + tt.sval);
                        } else {
                            tt.pushBack();
                        }
                    }
                    t.concatenate(new AffineTransform(m));
                    
                } else if (type.equals("translate")) {
                    double tx, ty;
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("X-translation value not found in transform "+str);
                    }
                    tx = tt.nval;
                    if (tt.nextToken() == StreamTokenizer.TT_WORD &&
                            (tt.sval.startsWith("E") || tt.sval.startsWith("e"))) {
                        double mantissa = tt.nval;
                        tx = Double.valueOf(tx + tt.sval);
                    } else {
                        tt.pushBack();
                    }
                    if (tt.nextToken() == StreamTokenizer.TT_NUMBER) {
                        ty = tt.nval;
                        if (tt.nextToken() == StreamTokenizer.TT_WORD &&
                                (tt.sval.startsWith("E") || tt.sval.startsWith("e"))) {
                            double mantissa = tt.nval;
                            ty = Double.valueOf(ty + tt.sval);
                        } else {
                            tt.pushBack();
                        }
                    } else {
                        tt.pushBack();
                        ty = 0;
                    }
                    t.translate(tx, ty);
                    
                } else if (type.equals("scale")) {
                    double sx, sy;
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("X-scale value not found in transform "+str);
                    }
                    sx = tt.nval;
                    if (tt.nextToken() == StreamTokenizer.TT_WORD &&
                            (tt.sval.startsWith("E") || tt.sval.startsWith("e"))) {
                        double mantissa = tt.nval;
                        sx = Double.valueOf(sx + tt.sval);
                    } else {
                        tt.pushBack();
                    }
                    if (tt.nextToken() == StreamTokenizer.TT_NUMBER) {
                        sy = tt.nval;
                        if (tt.nextToken() == StreamTokenizer.TT_WORD &&
                                (tt.sval.startsWith("E") || tt.sval.startsWith("e"))) {
                            double mantissa = tt.nval;
                            sy = Double.valueOf(sy + tt.sval);
                        } else {
                            tt.pushBack();
                        }
                    } else {
                        tt.pushBack();
                        sy = sx;
                    }
                    t.scale(sx, sy);
                    
                } else if (type.equals("rotate")) {
                    double angle, cx, cy;
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("Angle value not found in transform "+str);
                    }
                    angle = tt.nval;
                    if (tt.nextToken() == StreamTokenizer.TT_WORD &&
                            (tt.sval.startsWith("E") || tt.sval.startsWith("e"))) {
                        double mantissa = tt.nval;
                        angle = Double.valueOf(angle + tt.sval);
                    } else {
                        tt.pushBack();
                    }
                    if (tt.nextToken() == StreamTokenizer.TT_NUMBER) {
                        cx = tt.nval;
                        if (tt.nextToken() == StreamTokenizer.TT_WORD &&
                                (tt.sval.startsWith("E") || tt.sval.startsWith("e"))) {
                            double mantissa = tt.nval;
                            cx = Double.valueOf(cx + tt.sval);
                        } else {
                            tt.pushBack();
                        }
                        if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                            throw new IOException("Y-center value not found in transform "+str);
                        }
                        cy = tt.nval;
                        if (tt.nextToken() == StreamTokenizer.TT_WORD &&
                                (tt.sval.startsWith("E") || tt.sval.startsWith("e"))) {
                            double mantissa = tt.nval;
                            cy = Double.valueOf(cy + tt.sval);
                        } else {
                            tt.pushBack();
                        }
                    } else {
                        tt.pushBack();
                        cx = cy = 0;
                    }
                    t.rotate(angle * Math.PI / 180d, cx * Math.PI / 180d, cy * Math.PI / 180d);
                    
                    
                } else if (type.equals("skewX")) {
                    double angle;
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("Skew angle not found in transform "+str);
                    }
                    angle = tt.nval;
                    if (tt.nextToken() == StreamTokenizer.TT_WORD &&
                            (tt.sval.startsWith("E") || tt.sval.startsWith("e"))) {
                        double mantissa = tt.nval;
                        angle = Double.valueOf(angle + tt.sval);
                    } else {
                        tt.pushBack();
                    }
                    t.concatenate(new AffineTransform(
                            1, 0, Math.tan(angle * Math.PI / 180), 1, 0, 0
                            ));
                    
                } else if (type.equals("skewY")) {
                    double angle;
                    if (tt.nextToken() != StreamTokenizer.TT_NUMBER) {
                        throw new IOException("Skew angle not found in transform "+str);
                    }
                    angle = tt.nval;
                    if (tt.nextToken() == StreamTokenizer.TT_WORD &&
                            (tt.sval.startsWith("E") || tt.sval.startsWith("e"))) {
                        double mantissa = tt.nval;
                        angle = Double.valueOf(angle + tt.sval);
                    } else {
                        tt.pushBack();
                    }
                    t.concatenate(new AffineTransform(
                            1, Math.tan(angle * Math.PI / 180), 0, 1, 0, 0
                            ));
                    
                }else {
                    throw new IOException("Unknown transform "+type+" in "+str);
                }
                if (tt.nextToken() != ')') {
                    throw new IOException("')' not found in transform "+str);
                }
            }
        }
        return t;
    }
    
    public javax.swing.filechooser.FileFilter getFileFilter() {
        return new ExtensionFileFilter("Scalable Vector Graphics (SVG)", "svg");
    }
    
    public JComponent getInputFormatAccessory() {
        return null;
    }
    
    public void read(File file, Drawing drawing) throws IOException {
        this.url = file.toURL();
        BufferedInputStream in = null;
        try {
            in = new BufferedInputStream(new FileInputStream(file));
            read(in, drawing);
        } finally {
            if (in != null) {
                in.close();
            }
        }
        this.url = null;
    }
    
    public void read(InputStream in, Drawing drawing) throws IOException {
        drawing.addAll(readFigures(in));
    }
    
    public boolean isDataFlavorSupported(DataFlavor flavor) {
        return flavor.getPrimaryType().equals("image") &&
                flavor.getSubType().equals("svg+xml");
    }
    
    public LinkedList<Figure> readFigures(Transferable t) throws UnsupportedFlavorException, IOException {
        InputStream in = null;
        try {
            in = (InputStream) t.getTransferData(new DataFlavor("image/svg+xml", "Image SVG"));
            return readFigures(in);
        } finally {
            if (in != null) { in.close(); }
        }
    }
}
